;==============================================================================

; File: "_num.scm", Time-stamp: <2005-07-07 22:47:44 feeley>

; Copyright (C) 1994-2005 by Marc Feeley, All Rights Reserved.
; Copyright (C) 2004-2005 by Brad Lucier, All Rights Reserved.

;==============================================================================

(##include "header.scm")
(c-declare "#include \"mem.h\"")
(##define-macro (use-fast-bignum-algorithms) #t)

;==============================================================================

; Implementation of exceptions.

(implement-library-type-range-exception)

(define-prim (##raise-range-exception arg-num proc . args)
  (##extract-procedure-and-arguments
   proc
   args
   arg-num
   #f
   #f
   (lambda (procedure arguments arg-num dummy1 dummy2)
     (macro-raise
      (macro-make-range-exception procedure arguments arg-num)))))

(implement-library-type-divide-by-zero-exception)

(define-prim (##raise-divide-by-zero-exception proc . args)
  (##extract-procedure-and-arguments
   proc
   args
   #f
   #f
   #f
   (lambda (procedure arguments dummy1 dummy2 dummy3)
     (macro-raise
      (macro-make-divide-by-zero-exception procedure arguments)))))

;------------------------------------------------------------------------------

; Define type checking procedures.

(define-fail-check-type exact-signed-int8 'exact-signed-int8)
(define-fail-check-type exact-signed-int8-list 'exact-signed-int8-list)
(define-fail-check-type exact-unsigned-int8 'exact-unsigned-int8)
(define-fail-check-type exact-unsigned-int8-list 'exact-unsigned-int8-list)
(define-fail-check-type exact-signed-int16 'exact-signed-int16)
(define-fail-check-type exact-signed-int16-list 'exact-signed-int16-list)
(define-fail-check-type exact-unsigned-int16 'exact-unsigned-int16)
(define-fail-check-type exact-unsigned-int16-list 'exact-unsigned-int16-list)
(define-fail-check-type exact-signed-int32 'exact-signed-int32)
(define-fail-check-type exact-signed-int32-list 'exact-signed-int32-list)
(define-fail-check-type exact-unsigned-int32 'exact-unsigned-int32)
(define-fail-check-type exact-unsigned-int32-list 'exact-unsigned-int32-list)
(define-fail-check-type exact-signed-int64 'exact-signed-int64)
(define-fail-check-type exact-signed-int64-list 'exact-signed-int64-list)
(define-fail-check-type exact-unsigned-int64 'exact-unsigned-int64)
(define-fail-check-type exact-unsigned-int64-list 'exact-unsigned-int64-list)
(define-fail-check-type inexact-real 'inexact-real)
(define-fail-check-type inexact-real-list 'inexact-real-list)
(define-fail-check-type number 'number)
(define-fail-check-type real 'real)
(define-fail-check-type finite-real 'finite-real)
(define-fail-check-type rational 'rational)
(define-fail-check-type integer 'integer)
(define-fail-check-type exact-integer 'exact-integer)

;------------------------------------------------------------------------------

; Numerical type predicates.

(define-prim (##complex? x)
  (macro-number-dispatch x #f
    #t ; x = fixnum
    #t ; x = bignum
    #t ; x = ratnum
    #t ; x = flonum
    #t)) ; x = cpxnum

(define-prim (number? x)
  (macro-force-vars (x)
    (##complex? x)))

(define-prim (complex? x)
  (macro-force-vars (x)
    (##complex? x)))

(define-prim (##real? x)
  (macro-number-dispatch x #f
    #t ; x = fixnum
    #t ; x = bignum
    #t ; x = ratnum
    #t ; x = flonum
    (macro-cpxnum-real? x))) ; x = cpxnum

(define-prim (real? x)
  (macro-force-vars (x)
    (##real? x)))

(define-prim (##rational? x)
  (macro-number-dispatch x #f
    #t ; x = fixnum
    #t ; x = bignum
    #t ; x = ratnum
    (macro-flonum-rational? x) ; x = flonum
    (macro-cpxnum-rational? x))) ; x = cpxnum

(define-prim (rational? x)
  (macro-force-vars (x)
    (##rational? x)))

(define-prim (##integer? x)
  (macro-number-dispatch x #f
    #t ; x = fixnum
    #t ; x = bignum
    #f ; x = ratnum
    (macro-flonum-int? x) ; x = flonum
    (macro-cpxnum-int? x))) ; x = cpxnum

(define-prim (integer? x)
  (macro-force-vars (x)
    (##integer? x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Exactness predicates.

(define-prim (##exact? x)

  (define (type-error) #f)

  (macro-number-dispatch x (type-error)
    #t ; x = fixnum
    #t ; x = bignum
    #t ; x = ratnum
    #f ; x = flonum
    (and (##not (##flonum? (macro-cpxnum-real x))) ; x = cpxnum
         (##not (##flonum? (macro-cpxnum-imag x))))))

(define-prim (exact? x)
  (macro-force-vars (x)
    (let ()

      (define (type-error)
        (##fail-check-number 1 exact? x))

      (macro-number-dispatch x (type-error)
        #t ; x = fixnum
        #t ; x = bignum
        #t ; x = ratnum
        #f ; x = flonum
        (and (##not (##flonum? (macro-cpxnum-real x))) ; x = cpxnum
             (##not (##flonum? (macro-cpxnum-imag x))))))))

(define-prim (inexact? x)
  (macro-force-vars (x)
    (let ()

      (define (type-error)
        (##fail-check-number 1 inexact? x))

      (macro-number-dispatch x (type-error)
        #f ; x = fixnum
        #f ; x = bignum
        #f ; x = ratnum
        #t ; x = flonum
        (or (##flonum? (macro-cpxnum-real x)) ; x = cpxnum
            (##flonum? (macro-cpxnum-imag x)))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Comparison predicates.

(define-prim (##= x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (macro-number-dispatch x (type-error-on-x)

    (macro-number-dispatch y (type-error-on-y) ; x = fixnum
      (##fixnum.= x y)
      #f
      #f
      (if (##flonum.<-fixnum-exact? x)
        (##flonum.= (##flonum.<-fixnum x) y)
        (and (##flonum.finite? y)
             (##ratnum.= (##ratnum.<-exact-int x) (##flonum.->ratnum y))))
      (##cpxnum.= (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = bignum
      #f
      (##bignum.= x y)
      #f
      (and (##flonum.finite? y)
           (##ratnum.= (##ratnum.<-exact-int x) (##flonum.->ratnum y)))
      (##cpxnum.= (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = ratnum
      #f
      #f
      (##ratnum.= x y)
      (and (##flonum.finite? y)
           (##ratnum.= x (##flonum.->ratnum y)))
      (##cpxnum.= (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = flonum
      (if (##flonum.<-fixnum-exact? y)
        (##flonum.= x (##flonum.<-fixnum y))
        (and (##flonum.finite? x)
             (##ratnum.= (##flonum.->ratnum x) (##ratnum.<-exact-int y))))
      (and (##flonum.finite? x)
           (##ratnum.= (##flonum.->ratnum x) (##ratnum.<-exact-int y)))
      (and (##flonum.finite? x)
           (##ratnum.= (##flonum.->ratnum x) y))
      (##flonum.= x y)
      (##cpxnum.= (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = cpxnum
      (##cpxnum.= x (##cpxnum.<-noncpxnum y))
      (##cpxnum.= x (##cpxnum.<-noncpxnum y))
      (##cpxnum.= x (##cpxnum.<-noncpxnum y))
      (##cpxnum.= x (##cpxnum.<-noncpxnum y))
      (##cpxnum.= x y))))

(define-prim-nary-bool (= x y)
  #t
  (if (##complex? x) #t '(1))
  (##= x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-number))

(define-prim (##< x y #!optional (nan-result #f))

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (macro-number-dispatch x (type-error-on-x)

    (macro-number-dispatch y (type-error-on-y) ; x = fixnum
      (##fixnum.< x y)
      (##not (##bignum.negative? y))
      (##ratnum.< (##ratnum.<-exact-int x) y)
      (cond ((##flonum.finite? y)
             (if (##flonum.<-fixnum-exact? x)
               (##flonum.< (##flonum.<-fixnum x) y)
               (##ratnum.< (##ratnum.<-exact-int x) (##flonum.->ratnum y))))
            ((##flonum.nan? y)
             nan-result)
            (else
             (##flonum.positive? y)))
      (if (macro-cpxnum-real? y)
        (##< x (macro-cpxnum-real y) nan-result)
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = bignum
      (##bignum.negative? x)
      (##bignum.< x y)
      (##ratnum.< (##ratnum.<-exact-int x) y)
      (cond ((##flonum.finite? y)
             (##ratnum.< (##ratnum.<-exact-int x) (##flonum.->ratnum y)))
            ((##flonum.nan? y)
             nan-result)
            (else
             (##flonum.positive? y)))
      (if (macro-cpxnum-real? y)
        (##< x (macro-cpxnum-real y) nan-result)
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = ratnum
      (##ratnum.< x (##ratnum.<-exact-int y))
      (##ratnum.< x (##ratnum.<-exact-int y))
      (##ratnum.< x y)
      (cond ((##flonum.finite? y)
             (##ratnum.< x (##flonum.->ratnum y)))
            ((##flonum.nan? y)
             nan-result)
            (else
             (##flonum.positive? y)))
      (if (macro-cpxnum-real? y)
        (##< x (macro-cpxnum-real y) nan-result)
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = flonum
      (cond ((##flonum.finite? x)
             (if (##flonum.<-fixnum-exact? y)
               (##flonum.< x (##flonum.<-fixnum y))
               (##ratnum.< (##flonum.->ratnum x) (##ratnum.<-exact-int y))))
            ((##flonum.nan? x)
             nan-result)
            (else
             (##flonum.negative? x)))
      (cond ((##flonum.finite? x)
             (##ratnum.< (##flonum.->ratnum x) (##ratnum.<-exact-int y)))
            ((##flonum.nan? x)
             nan-result)
            (else
             (##flonum.negative? x)))
      (cond ((##flonum.finite? x)
             (##ratnum.< (##flonum.->ratnum x) y))
            ((##flonum.nan? x)
             nan-result)
            (else
             (##flonum.negative? x)))
      (if (or (##flonum.nan? x) (##flonum.nan? y))
        nan-result
        (##flonum.< x y))
      (if (macro-cpxnum-real? y)
        (##< x (macro-cpxnum-real y) nan-result)
        (type-error-on-y)))

    (if (macro-cpxnum-real? x) ; x = cpxnum
      (macro-number-dispatch y (type-error-on-y)
        (##< (macro-cpxnum-real x) y nan-result)
        (##< (macro-cpxnum-real x) y nan-result)
        (##< (macro-cpxnum-real x) y nan-result)
        (##< (macro-cpxnum-real x) y nan-result)
        (if (macro-cpxnum-real? y)
          (##< (macro-cpxnum-real x) (macro-cpxnum-real y) nan-result)
          (type-error-on-y)))
      (type-error-on-x))))

(define-prim-nary-bool (< x y)
  #t
  (if (##real? x) #t '(1))
  (##< x y #f)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-real))

(define-prim-nary-bool (> x y)
  #t
  (if (##real? x) #t '(1))
  (##< y x #f)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-real))

(define-prim-nary-bool (<= x y)
  #t
  (if (##real? x) #t '(1))
  (##not (##< y x #t))
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-real))

(define-prim-nary-bool (>= x y)
  #t
  (if (##real? x) #t '(1))
  (##not (##< x y #t))
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-real))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Numerical property predicates.

(define-prim (##zero? x)

  (define (type-error)
    (##fail-check-number 1 zero? x))

  (macro-number-dispatch x (type-error)
    (##fixnum.zero? x)
    #f
    #f
    (##flonum.zero? x)
    (and (let ((imag (macro-cpxnum-imag x)))
           (and (##flonum? imag) (##flonum.zero? imag)))
         (let ((real (macro-cpxnum-real x)))
           (if (##fixnum? real)
             (##fixnum.zero? real)
             (and (##flonum? real) (##flonum.zero? real)))))))

(define-prim (zero? x)
  (macro-force-vars (x)
    (##zero? x)))

(define-prim (##positive? x)

  (define (type-error)
    (##fail-check-real 1 positive? x))

  (macro-number-dispatch x (type-error)
    (##fixnum.positive? x)
    (##not (##bignum.negative? x))
    (##positive? (macro-ratnum-numerator x))
    (##flonum.positive? x)
    (if (macro-cpxnum-real? x)
      (##positive? (macro-cpxnum-real x))
      (type-error))))

(define-prim (positive? x)
  (macro-force-vars (x)
    (##positive? x)))

(define-prim (##negative? x)

  (define (type-error)
    (##fail-check-real 1 negative? x))

  (macro-number-dispatch x (type-error)
    (##fixnum.negative? x)
    (##bignum.negative? x)
    (##negative? (macro-ratnum-numerator x))
    (##flonum.negative? x)
    (if (macro-cpxnum-real? x)
      (##negative? (macro-cpxnum-real x))
      (type-error))))

(define-prim (negative? x)
  (macro-force-vars (x)
    (##negative? x)))

(define-prim (##odd? x)

  (define (type-error)
    (##fail-check-integer 1 odd? x))

  (macro-number-dispatch x (type-error)
    (##fixnum.odd? x)
    (macro-bignum-odd? x)
    (type-error)
    (if (macro-flonum-int? x)
      (##odd? (##flonum.->exact-int x))
      (type-error))
    (if (macro-cpxnum-int? x)
      (##odd? (##inexact->exact (macro-cpxnum-real x)))
      (type-error))))

(define-prim (odd? x)
  (macro-force-vars (x)
    (##odd? x)))

(define-prim (##even? x)

  (define (type-error)
    (##fail-check-integer 1 even? x))

  (macro-number-dispatch x (type-error)
    (##not (##fixnum.odd? x))
    (##not (macro-bignum-odd? x))
    (type-error)
    (if (macro-flonum-int? x)
      (##even? (##flonum.->exact-int x))
      (type-error))
    (if (macro-cpxnum-int? x)
      (##even? (##inexact->exact (macro-cpxnum-real x)))
      (type-error))))

(define-prim (even? x)
  (macro-force-vars (x)
    (##even? x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Max and min.

(define-prim (##max x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (macro-number-dispatch x (type-error-on-x)

    (macro-number-dispatch y (type-error-on-y) ; x = fixnum
      (##fixnum.max x y)
      (if (##< x y) y x)
      (if (##< x y) y x)
      (##flonum.max (##flonum.<-fixnum x) y)
      (if (macro-cpxnum-real? y)
        (##max x (macro-cpxnum-real y))
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = bignum
      (if (##< x y) y x)
      (if (##< x y) y x)
      (if (##< x y) y x)
      (##flonum.max (##flonum.<-exact-int x) y)
      (if (macro-cpxnum-real? y)
        (##max x (macro-cpxnum-real y))
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = ratnum
      (if (##< x y) y x)
      (if (##< x y) y x)
      (if (##< x y) y x)
      (##flonum.max (##flonum.<-ratnum x) y)
      (if (macro-cpxnum-real? y)
        (##max x (macro-cpxnum-real y))
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = flonum
      (##flonum.max x (##flonum.<-fixnum y))
      (##flonum.max x (##flonum.<-exact-int y))
      (##flonum.max x (##flonum.<-ratnum y))
      (##flonum.max x y)
      (if (macro-cpxnum-real? y)
        (##max x (macro-cpxnum-real y))
        (type-error-on-y)))

    (if (macro-cpxnum-real? x) ; x = cpxnum
      (macro-number-dispatch y (type-error-on-y)
        (##max (macro-cpxnum-real x) y)
        (##max (macro-cpxnum-real x) y)
        (##max (macro-cpxnum-real x) y)
        (##max (macro-cpxnum-real x) y)
        (if (macro-cpxnum-real? y)
          (##max (macro-cpxnum-real x) (macro-cpxnum-real y))
          (type-error-on-y)))
      (type-error-on-x))))

(define-prim-nary (max x y)
  ()
  (if (##real? x) x '(1))
  (##max x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-real))

(define-prim (##min x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (macro-number-dispatch x (type-error-on-x)

    (macro-number-dispatch y (type-error-on-y) ; x = fixnum
      (##fixnum.min x y)
      (if (##< x y) x y)
      (if (##< x y) x y)
      (##flonum.min (##flonum.<-fixnum x) y)
      (if (macro-cpxnum-real? y)
        (##min x (macro-cpxnum-real y))
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = bignum
      (if (##< x y) x y)
      (if (##< x y) x y)
      (if (##< x y) x y)
      (##flonum.min (##flonum.<-exact-int x) y)
      (if (macro-cpxnum-real? y)
        (##min x (macro-cpxnum-real y))
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = ratnum
      (if (##< x y) x y)
      (if (##< x y) x y)
      (if (##< x y) x y)
      (##flonum.min (##flonum.<-ratnum x) y)
      (if (macro-cpxnum-real? y)
        (##min x (macro-cpxnum-real y))
        (type-error-on-y)))

    (macro-number-dispatch y (type-error-on-y) ; x = flonum
      (##flonum.min x (##flonum.<-fixnum y))
      (##flonum.min x (##flonum.<-exact-int y))
      (##flonum.min x (##flonum.<-ratnum y))
      (##flonum.min x y)
      (if (macro-cpxnum-real? y)
        (##min x (macro-cpxnum-real y))
        (type-error-on-y)))

    (if (macro-cpxnum-real? x) ; x = cpxnum
      (macro-number-dispatch y (type-error-on-y)
        (##min (macro-cpxnum-real x) y)
        (##min (macro-cpxnum-real x) y)
        (##min (macro-cpxnum-real x) y)
        (##min (macro-cpxnum-real x) y)
        (if (macro-cpxnum-real? y)
          (##min (macro-cpxnum-real x) (macro-cpxnum-real y))
          (type-error-on-y)))
      (type-error-on-x))))

(define-prim-nary (min x y)
  ()
  (if (##real? x) x '(1))
  (##min x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-real))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; +, *, -, /

(define-prim (##+ x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (macro-number-dispatch x (type-error-on-x)

    (macro-number-dispatch y (type-error-on-y) ; x = fixnum
      (if (##fixnum.negative? x)
        (if (##fixnum.negative? y)
          (let ((r (##fixnum.+ x y)))
            (if (##fixnum.negative? r)
              r
              (##bignum.+ (##bignum.<-fixnum r) ##bignum.2*min-fixnum)))
          (##fixnum.+ x y))
        (if (##fixnum.negative? y)
          (##fixnum.+ x y)
          (let ((r (##fixnum.+ x y)))
            (if (##fixnum.negative? r)
              (##bignum.- (##bignum.<-fixnum r) ##bignum.2*min-fixnum)
              r))))
      (if (##fixnum.zero? x)
        y
        (##bignum.+ (##bignum.<-fixnum x) y))
      (if (##fixnum.zero? x)
        y
        (##ratnum.+ (##ratnum.<-exact-int x) y))
      (if (and (macro-special-case-exact-zero?) (##fixnum.zero? x))
        y
        (##flonum.+ (##flonum.<-fixnum x) y))
      (##cpxnum.+ (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = bignum
      (if (##fixnum.zero? y)
        x
        (##bignum.+ x (##bignum.<-fixnum y)))
      (##bignum.+ x y)
      (##ratnum.+ (##ratnum.<-exact-int x) y)
      (##flonum.+ (##flonum.<-exact-int x) y)
      (##cpxnum.+ (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = ratnum
      (if (##fixnum.zero? y)
        x
        (##ratnum.+ x (##ratnum.<-exact-int y)))
      (##ratnum.+ x (##ratnum.<-exact-int y))
      (##ratnum.+ x y)
      (##flonum.+ (##flonum.<-ratnum x) y)
      (##cpxnum.+ (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = flonum
      (if (and (macro-special-case-exact-zero?) (##fixnum.zero? y))
        x
        (##flonum.+ x (##flonum.<-fixnum y)))
      (##flonum.+ x (##flonum.<-exact-int y))
      (##flonum.+ x (##flonum.<-ratnum y))
      (##flonum.+ x y)
      (##cpxnum.+ (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = cpxnum
      (##cpxnum.+ x (##cpxnum.<-noncpxnum y))
      (##cpxnum.+ x (##cpxnum.<-noncpxnum y))
      (##cpxnum.+ x (##cpxnum.<-noncpxnum y))
      (##cpxnum.+ x (##cpxnum.<-noncpxnum y))
      (##cpxnum.+ x y))))

(define-prim-nary (+ x y)
  0
  (if (##complex? x) x '(1))
  (##+ x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-number))

(define-prim (##* x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (macro-number-dispatch x (type-error-on-x)

    (macro-number-dispatch y (type-error-on-y) ; x = fixnum
      (if (or (##fixnum.< x ##bignum.min-fixnum-div-mdigit-base)
              (##fixnum.< ##bignum.max-fixnum-div-mdigit-base x)
              (##not (##fixnum.< ##bignum.minus-mdigit-base y))
              (##fixnum.< ##bignum.mdigit-base y))
        (cond ((or (##fixnum.zero? x)
                   (##fixnum.zero? y))
               0)
              ((##fixnum.= x 1)
               y)
              ((##fixnum.= y 1)
               x)
              ((##fixnum.= x -1)
               (##negate y))
              ((##fixnum.= y -1)
               (##negate x))
              (else
               (##bignum.* (##bignum.<-fixnum x) (##bignum.<-fixnum y))))
        (##fixnum.* x y))
      (cond ((##fixnum.zero? x)
             0)
            ((##fixnum.= x 1)
             y)
            ((##fixnum.= x -1)
             (##negate y))
            (else
             (##bignum.* (##bignum.<-fixnum x) y)))
      (cond ((##fixnum.zero? x)
             0)
            ((##fixnum.= x 1)
             y)
            ((##fixnum.= x -1)
             (##negate y))
            (else
             (##ratnum.* (##ratnum.<-exact-int x) y)))
      (if (and (macro-special-case-exact-zero?) (##fixnum.zero? x))
        x
        (##flonum.* (##flonum.<-fixnum x) y))
      (##cpxnum.* (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = bignum
      (cond ((##eq? y 0)
             0)
            ((##eq? y 1)
             x)
            ((##eq? y -1)
             (##negate x))
            (else
             (##bignum.* x (##bignum.<-fixnum y))))
      (##bignum.* x y)
      (##ratnum.* (##ratnum.<-exact-int x) y)
      (##flonum.* (##flonum.<-exact-int x) y)
      (##cpxnum.* (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = ratnum
      (cond ((##fixnum.zero? y)
             0)
            ((##fixnum.= y 1)
             x)
            ((##fixnum.= y -1)
             (##negate x))
            (else
             (##ratnum.* x (##ratnum.<-exact-int y))))
      (##ratnum.* x (##ratnum.<-exact-int y))
      (##ratnum.* x y)
      (##flonum.* (##flonum.<-ratnum x) y)
      (##cpxnum.* (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = flonum
      (if (and (macro-special-case-exact-zero?) (##fixnum.zero? y))
        y
        (##flonum.* x (##flonum.<-fixnum y)))
      (##flonum.* x (##flonum.<-exact-int y))
      (##flonum.* x (##flonum.<-ratnum y))
      (##flonum.* x y)
      (##cpxnum.* (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = cpxnum
      (##cpxnum.* x (##cpxnum.<-noncpxnum y))
      (##cpxnum.* x (##cpxnum.<-noncpxnum y))
      (##cpxnum.* x (##cpxnum.<-noncpxnum y))
      (##cpxnum.* x (##cpxnum.<-noncpxnum y))
      (##cpxnum.* x y))))

(define-prim-nary (* x y)
  1
  (if (##complex? x) x '(1))
  (##* x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-number))

(define-prim (##negate x)

  (##define-macro (type-error) `'(1))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.negative? x)
      (let ((r (##fixnum.- x)))
        (if (##fixnum.negative? r)
          (##bignum.- (##bignum.<-fixnum r) ##bignum.2*min-fixnum)
          r))
      (##fixnum.- x))
    (##bignum.- (##bignum.<-fixnum 0) x)
    (macro-ratnum-make (##negate (macro-ratnum-numerator x))
                       (macro-ratnum-denominator x))
    (##flonum.- x)
    (##make-rectangular (##negate (macro-cpxnum-real x))
                        (##negate (macro-cpxnum-imag x)))))

(define-prim (##- x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (macro-number-dispatch x (type-error-on-x)

    (macro-number-dispatch y (type-error-on-y) ; x = fixnum
      (if (##fixnum.negative? x)
        (if (##fixnum.negative? y)
          (##fixnum.- x y)
          (let ((r (##fixnum.- x y)))
            (if (##fixnum.negative? r)
              r
              (##bignum.+ (##bignum.<-fixnum r) ##bignum.2*min-fixnum))))
        (if (##fixnum.negative? y)
          (let ((r (##fixnum.- x y)))
            (if (##fixnum.negative? r)
              (##bignum.- (##bignum.<-fixnum r) ##bignum.2*min-fixnum)
              r))
          (##fixnum.- x y)))
      (##bignum.- (##bignum.<-fixnum x) y)
      (if (##fixnum.zero? x)
        (##negate y)
        (##ratnum.- (##ratnum.<-exact-int x) y))
      (if (and (macro-special-case-exact-zero?) (##fixnum.zero? x))
        (##flonum.- y)
        (##flonum.- (##flonum.<-fixnum x) y))
      (##cpxnum.- (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = bignum
      (if (##fixnum.zero? y)
        x
        (##bignum.- x (##bignum.<-fixnum y)))
      (##bignum.- x y)
      (##ratnum.- (##ratnum.<-exact-int x) y)
      (##flonum.- (##flonum.<-exact-int x) y)
      (##cpxnum.- (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = ratnum
      (if (##fixnum.zero? y)
        x
        (##ratnum.- x (##ratnum.<-exact-int y)))
      (##ratnum.- x (##ratnum.<-exact-int y))
      (##ratnum.- x y)
      (##flonum.- (##flonum.<-ratnum x) y)
      (##cpxnum.- (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = flonum
      (if (and (macro-special-case-exact-zero?) (##fixnum.zero? y))
        x
        (##flonum.- x (##flonum.<-fixnum y)))
      (##flonum.- x (##flonum.<-exact-int y))
      (##flonum.- x (##flonum.<-ratnum y))
      (##flonum.- x y)
      (##cpxnum.- (##cpxnum.<-noncpxnum x) y))

    (macro-number-dispatch y (type-error-on-y) ; x = cpxnum
      (##cpxnum.- x (##cpxnum.<-noncpxnum y))
      (##cpxnum.- x (##cpxnum.<-noncpxnum y))
      (##cpxnum.- x (##cpxnum.<-noncpxnum y))
      (##cpxnum.- x (##cpxnum.<-noncpxnum y))
      (##cpxnum.- x y))))

(define-prim-nary (- x y)
  ()
  (##negate x)
  (##- x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-number))

(define-prim (##inverse x)

  (##define-macro (type-error) `'(1))

  (define (divide-by-zero-error) #f)

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      (divide-by-zero-error)
      (if (##fixnum.negative? x)
        (if (##fixnum.= x -1)
          x
          (macro-ratnum-make -1 (##negate x)))
        (if (##fixnum.= x 1)
          x
          (macro-ratnum-make 1 x))))
    (if (##bignum.negative? x)
      (macro-ratnum-make -1 (##negate x))
      (macro-ratnum-make 1 x))
    (let ((num (macro-ratnum-numerator x))
          (den (macro-ratnum-denominator x)))
      (cond ((##eq? num 1)
             den)
            ((##eq? num -1)
             (##negate den))
            (else
             (if (##negative? num)
               (macro-ratnum-make (##negate den) (##negate num))
               (macro-ratnum-make den num)))))
    (##flonum./ (macro-inexact-+1) x)
    (##cpxnum./ (##cpxnum.<-noncpxnum 1) x)))

(define-prim (##/ x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (define (divide-by-zero-error) #f)

  (macro-number-dispatch y (type-error-on-y)

    (macro-number-dispatch x (type-error-on-x) ; y = fixnum
      (cond ((##fixnum.zero? y)
             (divide-by-zero-error))
            ((##fixnum.= y 1)
             x)
            ((##fixnum.= y -1)
             (##negate x))
            ((##fixnum.zero? x)
             0)
            ((##fixnum.= x 1)
             (##inverse y))
            (else
             (##ratnum./ (##ratnum.<-exact-int x) (##ratnum.<-exact-int y))))
      (cond ((##fixnum.zero? y)
             (divide-by-zero-error))
            ((##fixnum.= y 1)
             x)
            ((##fixnum.= y -1)
             (##negate x))
            (else
             (##ratnum./ (##ratnum.<-exact-int x) (##ratnum.<-exact-int y))))
       (cond ((##fixnum.zero? y)
             (divide-by-zero-error))
            ((##fixnum.= y 1)
             x)
            ((##fixnum.= y -1)
             (##negate x))
            (else
             (##ratnum./ x (##ratnum.<-exact-int y))))
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##flonum./ x (##flonum.<-fixnum y)))
      (if (##fixnum.zero? y)
        (divide-by-zero-error)
        (##cpxnum./ x (##cpxnum.<-noncpxnum y))))

    (macro-number-dispatch x (type-error-on-x) ; y = bignum
      (cond ((##fixnum.zero? x)
             0)
            ((##fixnum.= x 1)
             (##inverse y))
            (else
             (##ratnum./ (##ratnum.<-exact-int x) (##ratnum.<-exact-int y))))
      (##ratnum./ (##ratnum.<-exact-int x) (##ratnum.<-exact-int y))
      (##ratnum./ x (##ratnum.<-exact-int y))
      (##flonum./ x (##flonum.<-exact-int y))
      (##cpxnum./ x (##cpxnum.<-noncpxnum y)))

    (macro-number-dispatch x (type-error-on-x) ; y = ratnum
      (cond ((##fixnum.zero? x)
             0)
            ((##fixnum.= x 1)
             (##inverse y))
            (else
             (##ratnum./ (##ratnum.<-exact-int x) y)))
      (##ratnum./ (##ratnum.<-exact-int x) y)
      (##ratnum./ x y)
      (##flonum./ x (##flonum.<-ratnum y))
      (##cpxnum./ x (##cpxnum.<-noncpxnum y)))

    (macro-number-dispatch x (type-error-on-x) ; y = flonum, no error possible
      (if (and (macro-special-case-exact-zero?) (##fixnum.zero? x))
        x
        (##flonum./ (##flonum.<-fixnum x) y))
      (##flonum./ (##flonum.<-exact-int x) y)
      (##flonum./ (##flonum.<-ratnum x) y)
      (##flonum./ x y)
      (##cpxnum./ x (##cpxnum.<-noncpxnum y)))

    (macro-number-dispatch x (type-error-on-x) ; y = cpxnum
      (##cpxnum./ (##cpxnum.<-noncpxnum x) y)
      (##cpxnum./ (##cpxnum.<-noncpxnum x) y)
      (##cpxnum./ (##cpxnum.<-noncpxnum x) y)
      (##cpxnum./ (##cpxnum.<-noncpxnum x) y)
      (##cpxnum./ x y))))

(define-prim-nary (/ x y)
  ()
  (##inverse x)
  (##/ x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-number)
  (##not ##raise-divide-by-zero-exception))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; abs

(define-prim (##abs x)

  (define (type-error)
    (##fail-check-real 1 abs x))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.negative? x) (##negate x) x)
    (if (##bignum.negative? x) (##negate x) x)
    (macro-ratnum-make (##abs (macro-ratnum-numerator x))
                       (macro-ratnum-denominator x))
    (##flonum.abs x)
    (if (macro-cpxnum-real? x)
      (##make-rectangular (##abs (macro-cpxnum-real x))
                          (##abs (macro-cpxnum-imag x)))
      (type-error))))

(define-prim (abs x)
  (macro-force-vars (x)
    (##abs x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; quotient, remainder, modulo

(define-prim (##quotient x y)

  (define (type-error-on-x)
    (##fail-check-integer 1 quotient x y))

  (define (type-error-on-y)
    (##fail-check-integer 2 quotient x y))

  (define (divide-by-zero-error)
    (##raise-divide-by-zero-exception quotient x y))

  (define (exact-quotient x y)
    (##car (##exact-int.div x y)))

  (define (inexact-quotient x y)
    (let ((exact-y (##inexact->exact y)))
      (if (##eq? exact-y 0)
        (divide-by-zero-error)
        (##exact->inexact
         (##quotient (##inexact->exact x) exact-y)))))

  (macro-number-dispatch y (type-error-on-y)

    (macro-number-dispatch x (type-error-on-x) ; y = fixnum
      (cond ((##fixnum.= y 0)
             (divide-by-zero-error))
            ((##fixnum.= y -1) ; (quotient ##min-fixnum -1) is a bignum
             (##negate x))
            (else
             (##fixnum.quotient x y)))
      (cond ((##fixnum.= y 0)
             (divide-by-zero-error))
            (else
             (exact-quotient x y)))
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (inexact-quotient x y)
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (inexact-quotient x y)
        (type-error-on-x)))

    (macro-number-dispatch x (type-error-on-x) ; y = bignum
      (exact-quotient x y)
      (exact-quotient x y)
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (inexact-quotient x y)
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (inexact-quotient x y)
        (type-error-on-x)))

    (type-error-on-y) ; y = ratnum

    (macro-number-dispatch x (type-error-on-x) ; y = flonum
      (if (macro-flonum-int? y)
        (inexact-quotient x y)
        (type-error-on-y))
      (if (macro-flonum-int? y)
        (inexact-quotient x y)
        (type-error-on-y))
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (if (macro-flonum-int? y)
          (inexact-quotient x y)
          (type-error-on-y))
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (if (macro-flonum-int? y)
          (inexact-quotient x y)
          (type-error-on-y))
        (type-error-on-x)))

    (if (macro-cpxnum-int? y) ; y = cpxnum
      (macro-number-dispatch x (type-error-on-x)
        (inexact-quotient x y)
        (inexact-quotient x y)
        (type-error-on-x)
        (if (macro-flonum-int? x)
          (inexact-quotient x y)
          (type-error-on-x))
        (if (macro-cpxnum-int? x)
          (inexact-quotient x y)
          (type-error-on-x)))
      (type-error-on-y))))

(define-prim (quotient x y)
  (macro-force-vars (x y)
    (##quotient x y)))

(define-prim (##remainder x y)

  (define (type-error-on-x)
    (##fail-check-integer 1 remainder x y))

  (define (type-error-on-y)
    (##fail-check-integer 2 remainder x y))

  (define (divide-by-zero-error)
    (##raise-divide-by-zero-exception remainder x y))

  (define (exact-remainder x y)
    (##cdr (##exact-int.div x y)))

  (define (inexact-remainder x y)
    (let ((exact-y (##inexact->exact y)))
      (if (##eq? exact-y 0)
        (divide-by-zero-error)
        (##exact->inexact
         (##remainder (##inexact->exact x) exact-y)))))

  (macro-number-dispatch y (type-error-on-y)

    (macro-number-dispatch x (type-error-on-x) ; y = fixnum
      (cond ((##fixnum.= y 0)
             (divide-by-zero-error))
            (else
             (##fixnum.remainder x y)))
      (cond ((##fixnum.= y 0)
             (divide-by-zero-error))
            (else
             (exact-remainder x y)))
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (inexact-remainder x y)
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (inexact-remainder x y)
        (type-error-on-x)))

    (macro-number-dispatch x (type-error-on-x) ; y = bignum
      (exact-remainder x y)
      (exact-remainder x y)
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (inexact-remainder x y)
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (inexact-remainder x y)
        (type-error-on-x)))

    (type-error-on-y) ; y = ratnum

    (macro-number-dispatch x (type-error-on-x) ; y = flonum
      (if (macro-flonum-int? y)
        (inexact-remainder x y)
        (type-error-on-y))
      (if (macro-flonum-int? y)
        (inexact-remainder x y)
        (type-error-on-y))
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (if (macro-flonum-int? y)
          (inexact-remainder x y)
          (type-error-on-y))
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (if (macro-flonum-int? y)
          (inexact-remainder x y)
          (type-error-on-y))
        (type-error-on-x)))

    (if (macro-cpxnum-int? y) ; y = cpxnum
      (macro-number-dispatch x (type-error-on-x)
        (inexact-remainder x y)
        (inexact-remainder x y)
        (type-error-on-x)
        (if (macro-flonum-int? x)
          (inexact-remainder x y)
          (type-error-on-x))
        (if (macro-cpxnum-int? x)
          (inexact-remainder x y)
          (type-error-on-x)))
      (type-error-on-y))))

(define-prim (remainder x y)
  (macro-force-vars (x y)
    (##remainder x y)))

(define-prim (##modulo x y)

  (define (type-error-on-x)
    (##fail-check-integer 1 modulo x y))

  (define (type-error-on-y)
    (##fail-check-integer 2 modulo x y))

  (define (divide-by-zero-error)
    (##raise-divide-by-zero-exception modulo x y))

  (define (exact-modulo x y)
    (let ((r (##cdr (##exact-int.div x y))))
      (if (##eq? r 0)
        0
        (if (##eq? (##negative? x) (##negative? y))
          r
          (##+ r y)))))

  (define (inexact-modulo x y)
    (let ((exact-y (##inexact->exact y)))
      (if (##eq? exact-y 0)
        (divide-by-zero-error)
        (##exact->inexact
         (##modulo (##inexact->exact x) exact-y)))))

  (macro-number-dispatch y (type-error-on-y)

    (macro-number-dispatch x (type-error-on-x) ; y = fixnum
      (cond ((##fixnum.= y 0)
             (divide-by-zero-error))
            (else
             (##fixnum.modulo x y)))
      (cond ((##fixnum.= y 0)
             (divide-by-zero-error))
            (else
             (exact-modulo x y)))
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (inexact-modulo x y)
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (inexact-modulo x y)
        (type-error-on-x)))

    (macro-number-dispatch x (type-error-on-x) ; y = bignum
      (exact-modulo x y)
      (exact-modulo x y)
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (inexact-modulo x y)
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (inexact-modulo x y)
        (type-error-on-x)))

    (type-error-on-y) ; y = ratnum

    (macro-number-dispatch x (type-error-on-x) ; y = flonum
      (if (macro-flonum-int? y)
        (inexact-modulo x y)
        (type-error-on-y))
      (if (macro-flonum-int? y)
        (inexact-modulo x y)
        (type-error-on-y))
      (type-error-on-x)
      (if (macro-flonum-int? x)
        (if (macro-flonum-int? y)
          (inexact-modulo x y)
          (type-error-on-y))
        (type-error-on-x))
      (if (macro-cpxnum-int? x)
        (if (macro-flonum-int? y)
          (inexact-modulo x y)
          (type-error-on-y))
        (type-error-on-x)))

    (if (macro-cpxnum-int? y) ; y = cpxnum
      (macro-number-dispatch x (type-error-on-x)
        (inexact-modulo x y)
        (inexact-modulo x y)
        (type-error-on-x)
        (if (macro-flonum-int? x)
          (inexact-modulo x y)
          (type-error-on-x))
        (if (macro-cpxnum-int? x)
          (inexact-modulo x y)
          (type-error-on-x)))
      (type-error-on-y))))

(define-prim (modulo x y)
  (macro-force-vars (x y)
    (##modulo x y)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; gcd, lcm

(define-prim (##gcd x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (define (##fast-gcd u v)

    ;; See the paper "Fast Reduction and Composition of Binary Quadratic Forms"
    ;; by Arnold Schoenhage.  His algorithm and proof are derived from, and
    ;; basically the same for, his Controlled Euclidean Descent algorithm for
    ;; gcd, which he has never published.  This algorithm has complexity log N
    ;; times a constant times the complexity of a multiplication of the
    ;; same size.  We don't use it until we get to about 6800 bits.  Note that
    ;; this is the same place that we start using FFT multiplication and fast
    ;; division with Newton's method for finding inverses.

    ;; assumes u and v are nonnegative exact ints

    (define (make-gcd-matrix A_11 A_12
                             A_21 A_22)
      (##vector A_11 A_12
                A_21 A_22))

    (define (gcd-matrix_11 A)
      (##vector-ref A 0))

    (define (gcd-matrix_12 A)
      (##vector-ref A 1))

    (define (gcd-matrix_21 A)
      (##vector-ref A 2))

    (define (gcd-matrix_22 A)
      (##vector-ref A 3))

    (define (make-gcd-vector v_1 v_2)
      (##vector v_1 v_2))

    (define (gcd-vector_1 v)
      (##vector-ref v 0))

    (define (gcd-vector_2 v)
      (##vector-ref v 1))

    (define gcd-matrix-identity '#(1 0
                                   0 1))

    (define (gcd-matrix-multiply A B)
      (cond ((##eq? A gcd-matrix-identity)
             B)
            ((##eq? B gcd-matrix-identity)
             A)
            (else
             (let ((A_11 (gcd-matrix_11 A)) (A_12 (gcd-matrix_12 A))
                   (A_21 (gcd-matrix_21 A)) (A_22 (gcd-matrix_22 A))
                   (B_11 (gcd-matrix_11 B)) (B_12 (gcd-matrix_12 B))
                   (B_21 (gcd-matrix_21 B)) (B_22 (gcd-matrix_22 B)))
               (make-gcd-matrix (##+ (##* A_11 B_11)
                                     (##* A_12 B_21))
                                (##+ (##* A_11 B_12)
                                     (##* A_12 B_22))
                                (##+ (##* A_21 B_11)
                                     (##* A_22 B_21))
                                (##+ (##* A_21 B_12)
                                     (##* A_22 B_22)))))))

    (define (gcd-matrix-multiply-strassen A B)
      ;; from http://mathworld.wolfram.com/StrassenFormulas.html
      (cond ((##eq? A gcd-matrix-identity)
             B)
            ((##eq? B gcd-matrix-identity)
             A)
            (else
             (let ((A_11 (gcd-matrix_11 A)) (A_12 (gcd-matrix_12 A))
                   (A_21 (gcd-matrix_21 A)) (A_22 (gcd-matrix_22 A))
                   (B_11 (gcd-matrix_11 B)) (B_12 (gcd-matrix_12 B))
                   (B_21 (gcd-matrix_21 B)) (B_22 (gcd-matrix_22 B)))
               (let ((Q_1 (##* (##+ A_11 A_22) (##+ B_11 B_22)))
                     (Q_2 (##* (##+ A_21 A_22) B_11))
                     (Q_3 (##* A_11 (##- B_12 B_22)))
                     (Q_4 (##* A_22 (##- B_21 B_11)))
                     (Q_5 (##* (##+ A_11 A_12) B_22))
                     (Q_6 (##* (##- A_21 A_11) (##+ B_11 B_12)))
                     (Q_7 (##* (##- A_12 A_22) (##+ B_21 B_22))))
                 (make-gcd-matrix (##+ (##+ Q_1 Q_4) (##- Q_7 Q_5))
                                  (##+ Q_3 Q_5)
                                  (##+ Q_2 Q_4)
                                  (##+ (##+ Q_1 Q_3) (##- Q_6 Q_2))))))))

    (define (gcd-matrix-solve A y)
      (let ((y_1 (gcd-vector_1 y))
            (y_2 (gcd-vector_2 y)))
        (make-gcd-vector (##- (##* y_1 (gcd-matrix_22 A))
                              (##* y_2 (gcd-matrix_12 A)))
                         (##- (##* y_2 (gcd-matrix_11 A))
                              (##* y_1 (gcd-matrix_21 A))))))

    (define (x>=2^n x n)
      (cond ((##eq? x 0)
             #f)
            ((and (##fixnum? x)
                  (##fixnum.<= n ##bignum.mdigit-width))
             (##fixnum.>= x (##fixnum.arithmetic-shift-left 1 n)))
            (else
             (let ((x (if (##fixnum? x) (##bignum.<-fixnum x) x)))
               (let loop ((i (##fixnum.- (##bignum.mdigit-length x) 1)))
                 (let ((digit (##bignum.mdigit-ref x i)))
                   (if (##fixnum.zero? digit)
                       (loop (##fixnum.- i 1))
                       (let ((words (##fixnum.quotient n ##bignum.mdigit-width)))
                         (or (##fixnum.> i words)
                             (and (##fixnum.= i words)
                                  (##fixnum.>= digit
                                               (##fixnum.arithmetic-shift-left
                                                1
                                                (##fixnum.remainder n ##bignum.mdigit-width)))))))))))))

    (define (determined-minimal? u v s)
      ;; assumes  2^s <= u , v; s>= 0 fixnum
      ;; returns #t if we can determine that |u-v|<2^s
      ;; at least one of u and v is a bignum
      (let ((u (if (##fixnum? u) (##bignum.<-fixnum u) u))
            (v (if (##fixnum? v) (##bignum.<-fixnum v) v)))
        (let ((u-length (##bignum.mdigit-length u)))
          (and (##fixnum.= u-length (##bignum.mdigit-length v))
               (let loop ((i (##fixnum.- u-length 1)))
                 (let ((v-digit (##bignum.mdigit-ref v i))
                       (u-digit (##bignum.mdigit-ref u i)))
                   (if (and (##fixnum.zero? u-digit)
                            (##fixnum.zero? v-digit))
                       (loop (##fixnum.- i 1))
                       (and (##fixnum.= (##fixnum.quotient s ##bignum.mdigit-width)
                                        i)
                            (##fixnum.< (##fixnum.max (##fixnum.- u-digit v-digit)
                                                      (##fixnum.- v-digit u-digit))
                                        (##fixnum.arithmetic-shift-left
                                         1
                                         (##fixnum.remainder s ##bignum.mdigit-width)))))))))))

    (define (gcd-small-step cont M u v s)
      ;;  u, v >= 2^s
      ;; M is the matrix product of the partial sums of
      ;; the continued fraction representation of a/b so far
      ;; returns updated M, u, v, and a truth value
      ;;  u, v >= 2^s and
      ;; if last return value is #t, we know that
      ;; (- (max u v) (min u v)) < 2^s, i.e, u, v are minimal above 2^s

      (define (gcd-matrix-multiply-low M q)
        (let ((M_11 (gcd-matrix_11 M))
              (M_12 (gcd-matrix_12 M))
              (M_21 (gcd-matrix_21 M))
              (M_22 (gcd-matrix_22 M)))
          (make-gcd-matrix (##+ M_11 (##* q M_12))  M_12
                           (##+ M_21 (##* q M_22))  M_22)))

      (define (gcd-matrix-multiply-high M q)
        (let ((M_11 (gcd-matrix_11 M))
              (M_12 (gcd-matrix_12 M))
              (M_21 (gcd-matrix_21 M))
              (M_22 (gcd-matrix_22 M)))
          (make-gcd-matrix M_11  (##+ (##* q M_11) M_12)
                           M_21  (##+ (##* q M_21) M_22))))

      (if (or (##bignum? u)
              (##bignum? v))

          ;; if u and v are nearly equal bignums, the two ##< following this condition could
          ;; take O(N) time to compute.  When this happens, however, it
          ;; will be likely that determined-minimal? will return true.

          (cond ((determined-minimal? u v s)
                 (cont M
                       u
                       v
                       #t))
                ((##< u v)
                 (let* ((qr (##exact-int.div v u))
                        (q (##car qr))
                        (r (##cdr qr)))
                   (cond ((x>=2^n r s)
                          (cont (gcd-matrix-multiply-low M q)
                                u
                                r
                                #f))
                         ((##eq? q 1)
                          (cont M
                                u
                                v
                                #t))
                         (else
                          (cont (gcd-matrix-multiply-low M (##- q 1))
                                u
                                (##+ r u)
                                #t)))))
                ((##< v u)
                 (let* ((qr (##exact-int.div u v))
                        (q (##car qr))
                        (r (##cdr qr)))
                   (cond ((x>=2^n r s)
                          (cont (gcd-matrix-multiply-high M q)
                                r
                                v
                                #f))
                         ((##eq? q 1)
                          (cont M
                                u
                                v
                                #t))
                         (else
                          (cont (gcd-matrix-multiply-high M (##- q 1))
                                (##+ r v)
                                v
                                #t)))))
                (else
                 (cont M
                       u
                       v
                       #t)))
          ;; here u and v are fixnums, so 2^s, which is <= u and v, is also a fixnum
          (let ((two^s (##fixnum.arithmetic-shift-left 1 s)))
            (if (##fixnum.< u v)
                (if (##fixnum.< (##fixnum.- v u) two^s)
                    (cont M
                          u
                          v
                          #t)
                    (let ((r (##fixnum.remainder v u))
                          (q (##fixnum.quotient  v u)))
                      (if (##fixnum.>= r two^s)
                          (cont (gcd-matrix-multiply-low M q)
                                u
                                r
                                #f)
                          ;; the case when q is one and the remainder is < two^s
                          ;; is covered in the first test
                          (cont (gcd-matrix-multiply-low M (##fixnum.- q 1))
                                u
                                (##fixnum.+ r u)
                                #t))))
                ;; here u >= v, but the case u = v is covered by the first test
                (if (##fixnum.< (##fixnum.- u v) two^s)
                    (cont M
                          u
                          v
                          #t)
                    (let ((r (##fixnum.remainder u v))
                          (q (##fixnum.quotient  u v)))
                      (if (##fixnum.>= r two^s)
                          (cont (gcd-matrix-multiply-high M q)
                                r
                                v
                                #f)
                          ;; the case when q is one and the remainder is < two^s
                          ;; is covered in the first test
                          (cont (gcd-matrix-multiply-high M (##fixnum.- q 1))
                                (##fixnum.+ r v)
                                v
                                #t))))))))

    (define (gcd-middle-step cont a b h m-prime cont-needs-M?)
      ((lambda (cont)
         (if (and (x>=2^n a h)
                  (x>=2^n b h))
             (MR cont a b h cont-needs-M?)
             (cont gcd-matrix-identity a b)))
       (lambda (M x y)
         (let loop ((M M)
                    (x x)
                    (y y))
           (if (or (x>=2^n x h)
                   (x>=2^n y h))
               ((lambda (cont) (gcd-small-step cont M x y m-prime))
                (lambda (M x y minimal?)
                  (if minimal?
                      (cont M x y)
                      (loop M x y))))
               ((lambda (cont) (MR cont x y m-prime cont-needs-M?))
                (lambda (M-prime alpha beta)
                  (cont (if cont-needs-M?
                            (if (##fixnum.> (##fixnum.- h m-prime) 1024)
                                ;; here we trade off 1 multiplication for 21 additions
                                (gcd-matrix-multiply-strassen M M-prime)
                                (gcd-matrix-multiply          M M-prime))
                            gcd-matrix-identity)
                        alpha
                        beta))))))))

    (define (MR cont a b m cont-needs-M?)
      ((lambda (cont)
         (if (and (x>=2^n a (##fixnum.+ m 2))
                  (x>=2^n b (##fixnum.+ m 2)))
             (let ((n (##fixnum.- (##fixnum.max (##integer-length a)
                                                (##integer-length b))
                                  m)))
               ((lambda (cont)
                  (if (##fixnum.<= m n)
                      (cont m 0)
                      (cont n (##fixnum.- (##fixnum.+ m 1) n))))
                (lambda (m-prime p)
                  (let ((h (##fixnum.+ m-prime (##fixnum.quotient n 2))))
                    (if (##fixnum.< 0 p)
                        (let ((a   (##arithmetic-shift a (##fixnum.- p)))
                              (b   (##arithmetic-shift b (##fixnum.- p)))
                              (a_0 (##extract-bit-field p 0 a))
                              (b_0 (##extract-bit-field p 0 b)))
                          ((lambda (cont)
                             (gcd-middle-step cont a b h m-prime #t))
                           (lambda (M alpha beta)
                             (let ((M-inverse-v_0 (gcd-matrix-solve M (make-gcd-vector a_0 b_0))))
                               (cont (if cont-needs-M? M gcd-matrix-identity)
                                     (##+ (##arithmetic-shift alpha p)
                                          (gcd-vector_1 M-inverse-v_0))
                                     (##+ (##arithmetic-shift beta p)
                                          (gcd-vector_2 M-inverse-v_0)))))))
                        (gcd-middle-step cont a b h m-prime cont-needs-M?))))))
             (cont gcd-matrix-identity
                   a
                   b)))
       (lambda (M alpha beta)
         (let loop ((M M)
                    (alpha alpha)
                    (beta beta)
                    (minimal? #f))
           (if minimal?
               (cont M alpha beta)
               (gcd-small-step loop M alpha beta m))))))

    ((lambda (cont)
       (if (and (use-fast-bignum-algorithms)
                (##bignum? u)
                (##bignum? v)
                (x>=2^n u ##bignum.fast-gcd-size)
                (x>=2^n v ##bignum.fast-gcd-size))
           (MR cont u v ##bignum.fast-gcd-size #f)
           (cont 0 u v)))
     (lambda (M a b)
       (general-base a b))))

  (define (general-base a b)
    (##declare (not interrupts-enabled))
    (if (##eq? b 0)
        a
        (if (##fixnum? b)
            (fixnum-base b (##remainder a b))
            (general-base b (##remainder a b)))))

  (define (fixnum-base a b)
    (##declare (not interrupts-enabled))
    (if (##eq? b 0)
        a
        (let ((a b)
              (b (##fixnum.remainder a b)))
          (if (##eq? b 0)
              a
              (fixnum-base b (##fixnum.remainder a b))))))

  (define (exact-gcd x y)
    (let ((x (##abs x))
          (y (##abs y)))
      (cond ((##eq? x 0)
             y)
            ((##eq? y 0)
             x)
            ((and (##fixnum? x) (##fixnum? y))
             (fixnum-base x y))
            (else
             (let ((x-first-bit (##first-set-bit x))
                   (y-first-bit (##first-set-bit y)))
               (##arithmetic-shift
                (##fast-gcd (##arithmetic-shift x (##fixnum.- x-first-bit))
                            (##arithmetic-shift y (##fixnum.- y-first-bit)))
                (##fixnum.min x-first-bit y-first-bit)))))))

  (define (inexact-gcd x y)
    (##exact->inexact
     (exact-gcd (##inexact->exact x)
                (##inexact->exact y))))

  (cond ((##not (##integer? x))
         (type-error-on-x))
        ((##not (##integer? y))
         (type-error-on-y))
        ((##eq? x y)
         (##abs x))
        (else
         (if (and (##exact? x) (##exact? y))
           (exact-gcd x y)
           (inexact-gcd x y)))))

(define-prim-nary (gcd x y)
  0
  (if (##integer? x) x '(1))
  (##gcd x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-integer))

(define-prim (##lcm x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (define (exact-lcm x y)
    (if (or (##eq? x 0) (##eq? y 0))
      0
      (##abs (##* (##quotient x (##gcd x y))
                  y))))

  (define (inexact-lcm x y)
    (##exact->inexact
     (exact-lcm (##inexact->exact x)
                (##inexact->exact y))))

  (cond ((##not (##integer? x))
         (type-error-on-x))
        ((##not (##integer? y))
         (type-error-on-y))
        (else
         (if (and (##exact? x) (##exact? y))
           (exact-lcm x y)
           (inexact-lcm x y)))))

(define-prim-nary (lcm x y)
  1
  (if (##integer? x) x '(1))
  (##lcm x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-integer))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; numerator, denominator

(define-prim (##numerator x)

  (define (type-error)
    (##fail-check-rational 1 numerator x))

  (macro-number-dispatch x (type-error)
    x
    x
    (macro-ratnum-numerator x)
    (cond ((##flonum.zero? x)
           x)
          ((macro-flonum-rational? x)
           (##exact->inexact (##numerator (##flonum.inexact->exact x))))
          (else
           (type-error)))
    (if (macro-cpxnum-rational? x)
      (##numerator (macro-cpxnum-real x))
      (type-error))))

(define-prim (numerator x)
  (macro-force-vars (x)
    (##numerator x)))

(define-prim (##denominator x)

  (define (type-error)
    (##fail-check-rational 1 denominator x))

  (macro-number-dispatch x (type-error)
    1
    1
    (macro-ratnum-denominator x)
    (if (macro-flonum-rational? x)
      (##exact->inexact (##denominator (##flonum.inexact->exact x)))
      (type-error))
    (if (macro-cpxnum-rational? x)
      (##denominator (macro-cpxnum-real x))
      (type-error))))

(define-prim (denominator x)
  (macro-force-vars (x)
    (##denominator x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; floor, ceiling, truncate, round

(define-prim (##floor x)

  (define (type-error)
    (##fail-check-finite-real 1 floor x))

  (macro-number-dispatch x (type-error)
    x
    x
    (let ((num (macro-ratnum-numerator x))
          (den (macro-ratnum-denominator x)))
      (if (##negative? num)
        (##quotient (##- num (##- den 1)) den)
        (##quotient num den)))
    (if (##flonum.finite? x)
      (##flonum.floor x)
      (type-error))
    (if (macro-cpxnum-real? x)
      (##floor (macro-cpxnum-real x))
      (type-error))))

(define-prim (floor x)
  (macro-force-vars (x)
    (##floor x)))

(define-prim (##ceiling x)

  (define (type-error)
    (##fail-check-finite-real 1 ceiling x))

  (macro-number-dispatch x (type-error)
    x
    x
    (let ((num (macro-ratnum-numerator x))
          (den (macro-ratnum-denominator x)))
      (if (##negative? num)
        (##quotient num den)
        (##quotient (##+ num (##- den 1)) den)))
    (if (##flonum.finite? x)
      (##flonum.ceiling x)
      (type-error))
    (if (macro-cpxnum-real? x)
      (##ceiling (macro-cpxnum-real x))
      (type-error))))

(define-prim (ceiling x)
  (macro-force-vars (x)
    (##ceiling x)))

(define-prim (##truncate x)

  (define (type-error)
    (##fail-check-finite-real 1 truncate x))

  (macro-number-dispatch x (type-error)
    x
    x
    (##quotient (macro-ratnum-numerator x)
                (macro-ratnum-denominator x))
    (if (##flonum.finite? x)
      (##flonum.truncate x)
      (type-error))
    (if (macro-cpxnum-real? x)
      (##truncate (macro-cpxnum-real x))
      (type-error))))

(define-prim (truncate x)
  (macro-force-vars (x)
    (##truncate x)))

(define-prim (##round x)

  (define (type-error)
    (##fail-check-finite-real 1 round x))

  (macro-number-dispatch x (type-error)
    x
    x
    (let ((num (macro-ratnum-numerator x))
          (den (macro-ratnum-denominator x)))
      (if (##eq? den 2)
        (##arithmetic-shift (##arithmetic-shift (##+ num 1) -2) 1)
        (##floor
         (##ratnum.normalize
          (##+ (##arithmetic-shift num 1) den)
          (##arithmetic-shift den 1)))))
    (if (##flonum.finite? x)
      (##flonum.round x)
      (type-error))
    (if (macro-cpxnum-real? x)
      (##round (macro-cpxnum-real x))
      (type-error))))

(define-prim (round x)
  (macro-force-vars (x)
    (##round x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; rationalize

(define-prim (##rationalize x y)

  (define (simplest-rational1 x y)
    (if (##< y x)
      (simplest-rational2 y x)
      (simplest-rational2 x y)))

  (define (simplest-rational2 x y)
    (cond ((##not (##< x y))
           x)
          ((##positive? x)
           (simplest-rational3 x y))
          ((##negative? y)
           (##negate (simplest-rational3 (##negate y) (##negate x))))
          (else
           0)))

  (define (simplest-rational3 x y)
    (let ((fx (##floor x))
          (fy (##floor y)))
      (cond ((##not (##< fx x))
             fx)
            ((##= fx fy)
             (##+ fx
                  (##inverse
                   (simplest-rational3
                    (##inverse (##- y fy))
                    (##inverse (##- x fx))))))
            (else
             (##+ fx 1)))))

  (cond ((##not (##rational? x))
         (##fail-check-finite-real 1 rationalize x y))
        ((and (##flonum? y)
              (##flonum.= y (macro-inexact-+inf)))
         (macro-inexact-+0))
        ((##not (##rational? y))
         (##fail-check-real 2 rationalize x y))
        ((##negative? y)
         (##raise-range-exception 2 rationalize x y))
        ((and (##exact? x) (##exact? y))
         (simplest-rational1 (##- x y) (##+ x y)))
        (else
         (let ((exact-x (##inexact->exact x))
               (exact-y (##inexact->exact y)))
           (##exact->inexact
            (simplest-rational1 (##- exact-x exact-y)
                                (##+ exact-x exact-y)))))))

(define-prim (rationalize x y)
  (macro-force-vars (x y)
    (##rationalize x y)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; trigonometry and complex numbers

(define-prim (##exp x)

  (define (type-error)
    (##fail-check-number 1 exp x))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      1
      (##flonum.exp (##flonum.<-fixnum x)))
    (##flonum.exp (##flonum.<-exact-int x))
    (##flonum.exp (##flonum.<-ratnum x))
    (##flonum.exp x)
    (##make-polar (##exp (macro-cpxnum-real x))
                  (macro-cpxnum-imag x))))

(define-prim (exp x)
  (macro-force-vars (x)
    (##exp x)))

(define-prim (##log x)

  (define (type-error)
    (##fail-check-number 1 log x))

  (define (range-error)
    (##raise-range-exception 1 log x))

  (define (negative-log x)
    (##make-rectangular (##log (##negate x)) (macro-inexact-+pi)))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      (range-error)
      (if (##fixnum.negative? x)
        (negative-log x)
        (if (##eq? x 1)
          0
          (##flonum.log (##flonum.<-fixnum x)))))
    (if (##bignum.negative? x)
      (negative-log x)
      (##flonum.log (##flonum.<-exact-int x)))
    (if (##negative? (macro-ratnum-numerator x))
      (negative-log x)
      (##flonum.log (##flonum.<-ratnum x)))
    (if (##flonum.nan? x)
      x
      (if (##flonum.negative? (##flonum.copysign (macro-inexact-+1) x))
        (negative-log x)
        (##flonum.log x)))
    (##make-rectangular (##log (##magnitude x)) (##angle x))))

(define-prim (log x)
  (macro-force-vars (x)
    (##log x)))

(define-prim (##sin x)

  (define (type-error)
    (##fail-check-number 1 sin x))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      0
      (##flonum.sin (##flonum.<-fixnum x)))
    (##flonum.sin (##flonum.<-exact-int x))
    (##flonum.sin (##flonum.<-ratnum x))
    (##flonum.sin x)
    (##/ (##- (##exp (##make-rectangular
                      (##negate (macro-cpxnum-imag x))
                      (macro-cpxnum-real x)))
              (##exp (##make-rectangular
                      (macro-cpxnum-imag x)
                      (##negate (macro-cpxnum-real x)))))
         (macro-cpxnum-+2i))))

(define-prim (sin x)
  (macro-force-vars (x)
    (##sin x)))

(define-prim (##cos x)

  (define (type-error)
    (##fail-check-number 1 cos x))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      1
      (##flonum.cos (##flonum.<-fixnum x)))
    (##flonum.cos (##flonum.<-exact-int x))
    (##flonum.cos (##flonum.<-ratnum x))
    (##flonum.cos x)
    (##/ (##+ (##exp (##make-rectangular
                      (##negate (macro-cpxnum-imag x))
                      (macro-cpxnum-real x)))
              (##exp (##make-rectangular
                      (macro-cpxnum-imag x)
                      (##negate (macro-cpxnum-real x)))))
         2)))

(define-prim (cos x)
  (macro-force-vars (x)
    (##cos x)))

(define-prim (##tan x)

  (define (type-error)
    (##fail-check-number 1 tan x))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      0
      (##flonum.tan (##flonum.<-fixnum x)))
    (##flonum.tan (##flonum.<-exact-int x))
    (##flonum.tan (##flonum.<-ratnum x))
    (##flonum.tan x)
    (let ((a (##exp (##make-rectangular
                     (##negate (macro-cpxnum-imag x))
                     (macro-cpxnum-real x))))
          (b (##exp (##make-rectangular
                     (macro-cpxnum-imag x)
                     (##negate (macro-cpxnum-real x))))))
      (let ((c (##/ (##- a b) (##+ a b))))
        (##make-rectangular (##imag-part c) (##negate (##real-part c)))))))

(define-prim (tan x)
  (macro-force-vars (x)
    (##tan x)))

(define-prim (##asin x)

  (define (type-error)
    (##fail-check-number 1 asin x))

  (define (safe-case x)
    (##* (macro-cpxnum--i)
         (##log (##+ (##* (macro-cpxnum-+i) x)
                     (##sqrt (##- 1 (##* x x)))))))

  (define (unsafe-case x)
    (##negate (safe-case (##negate x))))

  (define (real-case x)
    (cond ((##< x -1)
           (unsafe-case x))
          ((##< 1 x)
           (safe-case x))
          (else
           (##flonum.asin (##exact->inexact x)))))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      0
      (real-case x))
    (real-case x)
    (real-case x)
    (real-case x)
    (let ((imag (macro-cpxnum-imag x)))
      (if (or (##positive? imag)
              (and (##flonum? imag)
                   (##flonum.zero? imag)
                   (##negative? (macro-cpxnum-real x))))
        (unsafe-case x)
        (safe-case x)))))

(define-prim (asin x)
  (macro-force-vars (x)
    (##asin x)))

(define-prim (##acos x)

  (define (type-error)
    (##fail-check-number 1 acos x))

  (define (complex-case x)
    (##* (macro-cpxnum--i)
         (##log (##+ x
                     (##* (macro-cpxnum-+i) (##sqrt (##- 1 (##* x x))))))))

  (define (real-case x)
    (if (or (##< x -1) (##< 1 x))
      (complex-case x)
      (##flonum.acos (##exact->inexact x))))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      (macro-inexact-+pi/2)
      (real-case x))
    (real-case x)
    (real-case x)
    (real-case x)
    (complex-case x)))

(define-prim (acos x)
  (macro-force-vars (x)
    (##acos x)))

(define-prim (##atan x)

  (define (type-error)
    (##fail-check-number 1 atan x))

  (define (range-error)
    (##raise-range-exception 1 atan x))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.zero? x)
      0
      (##flonum.atan (##flonum.<-fixnum x)))
    (##flonum.atan (##flonum.<-exact-int x))
    (##flonum.atan (##flonum.<-ratnum x))
    (##flonum.atan x)
    (let ((real (macro-cpxnum-real x))
          (imag (macro-cpxnum-imag x)))
      (if (and (##eq? real 0) (##eq? imag 1))
        (range-error)
        (let ((a (##make-rectangular (##negate imag) real)))
          (##/ (##- (##log (##+ a 1)) (##log (##- 1 a)))
               (macro-cpxnum-+2i)))))))

(define-prim (##atan2 y x)
  (cond ((##not (##real? y))
         (##fail-check-real 1 atan y x))
        ((##not (##real? x))
         (##fail-check-real 2 atan y x))
        (else
         (##flonum.atan (##exact->inexact (##real-part y))
                        (##exact->inexact (##real-part x))))))

(define-prim (atan x #!optional (y (macro-absent-obj)))
  (macro-force-vars (x)
    (if (##eq? y (macro-absent-obj))
      (##atan x)
      (macro-force-vars (y)
        (##atan2 x y)))))

(define-prim (##sqrt x)

  (define (type-error)
    (##fail-check-number 1 sqrt x))

  (define (exact-int-sqrt x)
    (cond ((##eq? x 0)
           0)
          ((##negative? x)
           (##make-rectangular 0 (exact-int-sqrt (##negate x))))
          (else
           (let ((y (##exact-int.sqrt x)))
             (if (##eq? (##cdr y) 0)
               (##car y)
               (##flonum.sqrt (##flonum.<-exact-int x)))))))

  (macro-number-dispatch x (type-error)
    (exact-int-sqrt x)
    (exact-int-sqrt x)
    (##/ (exact-int-sqrt (macro-ratnum-numerator x))
         (exact-int-sqrt (macro-ratnum-denominator x)))
    (if (##flonum.negative? x)
      (##make-rectangular 0 (##flonum.sqrt (##flonum.- x)))
      (##flonum.sqrt x))
    (let ((real (##real-part x))
          (imag (##imag-part x)))
      (if (and (##flonum? imag)
               (##flonum.zero? imag))
        (if (##flonum.positive? (##flonum.copysign (macro-inexact-+1) imag))
          (cond ((##negative? real)
                 (##make-rectangular (macro-inexact-+0)
                                     (##exact->inexact
                                      (##sqrt (##negate real)))))
                ((and (##flonum? real)
                      (##flonum.nan? real))
                 (##make-rectangular real real))
                (else
                 (##make-rectangular (##exact->inexact (##sqrt real))
                                     (macro-inexact-+0))))
          (cond ((##negative? real)
                 (##make-rectangular (macro-inexact-+0)
                                     (##exact->inexact
                                      (##negate (##sqrt (##negate real))))))
                ((and (##flonum? real)
                      (##flonum.nan? real))
                 (##make-rectangular real real))
                (else
                 (##make-rectangular (##exact->inexact (##sqrt real))
                                     (macro-inexact--0)))))
        (##make-polar (##sqrt (##magnitude x))
                      (##/ (##angle x) 2))))))

(define-prim (sqrt x)
  (macro-force-vars (x)
    (##sqrt x)))

(define-prim (##expt x y)

  (define (exact-int-expt x y)

    ; x is a number, y is an exact-int >= 0

    (define (square x) (##* x x))

    (define (fixnum-expt-aux x y)
      ; x is a number, y is a fixnum >= 1
      (if (##fixnum.= y 1)
        x
        (let ((z
               (square
                (fixnum-expt-aux x (##fixnum.arithmetic-shift-right y 1)))))
          (if (##fixnum.even? y)
            z
            (##* x z)))))

    (define (exact-int-expt-aux x y)
      ; x is a number, y is an exact-int >= 1
      (if (##fixnum? y)
        (fixnum-expt-aux x y)
        (let ((z (square (exact-int-expt-aux x (##arithmetic-shift y -1)))))
          (if (##even? y)
            z
            (##* x z)))))

    (cond ((##fixnum? x)
           (cond ((##eq? y 0)
                  1)
                 ((##eq? x 0)
                  x)
                 ((##eq? x 1)
                  x)
                 (else
                  (exact-int-expt-aux x y))))
          ((##bignum? x)
           (if (##eq? y 0)
             1
             (exact-int-expt-aux x y)))
          ((##ratnum? x)
           (if (##eq? y 0)
             1
             (macro-ratnum-make
              (exact-int-expt-aux (macro-ratnum-numerator x) y)
              (exact-int-expt-aux (macro-ratnum-denominator x) y))))
          ((##flonum? x)
           (cond ((##flonum.nan? x)
                  x)
                 ((##eq? y 0)
                  1)
                 ((##flonum.zero? x)
                  (if (##even? y)
                    (macro-inexact-+0)
                    x))
                 (else
                  (exact-int-expt-aux x y))))
          (else
           (let ((r (macro-cpxnum-real x))
                 (i (macro-cpxnum-imag x)))
             (cond ((or (and (##flonum? r) (##flonum.nan? r))
                        (and (##flonum? i) (##flonum.nan? i)))
                    x)
                   ((##eq? y 0)
                    1)
                   (else
                    (exact-int-expt-aux x y)))))))

  (define (real-expt x y) ; y is not an exact-int and y >= 0
    (cond ((##zero? y)
           (macro-inexact-+1))
          ((##zero? x)
           (if (##eq? x 0) 0 (macro-inexact-+0)))
          ((##eq? x 1)
           1)
          ((and (##flonum? y) (macro-flonum-int? y))
           (let ((result (exact-int-expt x (##flonum.->exact-int y))))
             (if (##cpxnum? result)
               (let ((r (macro-cpxnum-real result))
                     (i (macro-cpxnum-imag result)))
                 (macro-cpxnum-make (if (##eq? r 0)
                                      0
                                      (##exact->inexact r))
                                    (##exact->inexact i)))
               (##exact->inexact result))))
          (else
           (complex-expt x y))))

  (define (complex-expt x y)
    (##exp (##* (##log x) y)))

  (define (invert z)
    (if (and (##flonum? z) (##flonum.nan? z))
      z
      (let ((result (##inverse z)))
        (if (##not result)
          (##raise-range-exception 1 expt x y)
          result))))

  (if (##complex? x)
    (macro-number-dispatch y (##fail-check-number 2 expt x y)
      (if (##fixnum.negative? y)
        (invert (exact-int-expt x (##negate y)))
        (exact-int-expt x y))
      (if (##bignum.negative? y)
        (invert (exact-int-expt x (##negate y)))
        (exact-int-expt x y))
      (if (and (##flonum? x) (##flonum.nan? x))
        x
        (if (##negative? (macro-ratnum-numerator y))
          (invert (real-expt x (##negate y)))
          (real-expt x y)))
      (if (and (##flonum? x) (##flonum.nan? x))
        x
        (if (##flonum.nan? y)
          y
          (if (##flonum.negative? y)
            (invert (real-expt x (##flonum.- y)))
            (real-expt x y))))
      (if (and (##flonum? x) (##flonum.nan? x))
        x
        (complex-expt x y)))
    (##fail-check-number 1 expt x y)))

(define-prim (expt x y)
  (macro-force-vars (x y)
    (##expt x y)))

(define-prim (##make-rectangular x y)
  (cond ((##not (##real? x))
         (##fail-check-real 1 make-rectangular x y))
        ((##not (##real? y))
         (##fail-check-real 2 make-rectangular x y))
        (else
         (let ((real (##real-part x))
               (imag (##real-part y)))
           (if (##eq? imag 0)
             real
             (macro-cpxnum-make real imag))))))

(define-prim (make-rectangular x y)
  (macro-force-vars (x y)
    (##make-rectangular x y)))

(define-prim (##make-polar x y)
  (cond ((##not (##real? x))
         (##fail-check-real 1 make-polar x y))
        ((##not (##real? y))
         (##fail-check-real 2 make-polar x y))
        (else
         (let ((real-x (##real-part x))
               (real-y (##real-part y)))
           (##make-rectangular (##* real-x (##cos real-y))
                               (##* real-x (##sin real-y)))))))

(define-prim (make-polar x y)
  (macro-force-vars (x y)
    (##make-polar x y)))

(define-prim (##real-part x)

  (define (type-error)
    (##fail-check-number 1 real-part x))

  (macro-number-dispatch x (type-error)
    x x x x (macro-cpxnum-real x)))

(define-prim (real-part x)
   (macro-force-vars (x)
     (##real-part x)))

(define-prim (##imag-part x)

  (define (type-error)
    (##fail-check-number 1 imag-part x))

  (macro-number-dispatch x (type-error)
    0 0 0 0 (macro-cpxnum-imag x)))

(define-prim (imag-part x)
  (macro-force-vars (x)
    (##imag-part x)))

(define-prim (##magnitude x)

  (define (type-error)
    (##fail-check-number 1 magnitude x))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.negative? x) (##negate x) x)
    (if (##bignum.negative? x) (##negate x) x)
    (macro-ratnum-make (##abs (macro-ratnum-numerator x))
                       (macro-ratnum-denominator x))
    (##flonum.abs x)
    (let ((abs-r (##abs (##real-part x)))
          (abs-i (##abs (##imag-part x))))

      (define (complex-magn a b)
        (cond ((##eq? a 0)
               b)
              ((and (##flonum? a) (##flonum.zero? a))
               (##exact->inexact b))
              (else
               (let ((c (##/ a b)))
                 (##* b (##sqrt (##+ (##* c c) 1)))))))

      (cond ((or (and (##flonum? abs-r)
                      (##flonum.= abs-r (macro-inexact-+inf)))
                 (and (##flonum? abs-i)
                      (##flonum.= abs-i (macro-inexact-+inf))))
             (macro-inexact-+inf))
            ((and (##flonum? abs-r) (##flonum.nan? abs-r))
             abs-r)
            ((and (##flonum? abs-i) (##flonum.nan? abs-i))
             abs-i)
            (else
             (if (##< abs-r abs-i)
               (complex-magn abs-r abs-i)
               (complex-magn abs-i abs-r)))))))

(define-prim (magnitude x)
  (macro-force-vars (x)
    (##magnitude x)))

(define-prim (##angle x)

  (define (type-error)
    (##fail-check-number 1 angle x))

  (macro-number-dispatch x (type-error)
    (if (##fixnum.negative? x)
      (macro-inexact-+pi)
      0)
    (if (##bignum.negative? x)
      (macro-inexact-+pi)
      0)
    (if (##negative? (macro-ratnum-numerator x))
      (macro-inexact-+pi)
      0)
    (if (##flonum.negative? (##flonum.copysign (macro-inexact-+1) x))
      (macro-inexact-+pi)
      (macro-inexact-+0))
    (##atan2 (macro-cpxnum-imag x) (macro-cpxnum-real x))))

(define-prim (angle x)
  (macro-force-vars (x)
    (##angle x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; exact->inexact, inexact->exact

(define-prim (##exact->inexact x)

  (define (type-error)
    (##fail-check-number 1 exact->inexact x))

  (macro-number-dispatch x (type-error)
    (##flonum.<-fixnum x)
    (##flonum.<-exact-int x)
    (##flonum.<-ratnum x)
    x
    (##make-rectangular (##exact->inexact (macro-cpxnum-real x))
                        (##exact->inexact (macro-cpxnum-imag x)))))

(define-prim (exact->inexact x)
  (macro-force-vars (x)
    (##exact->inexact x)))

(define-prim (##inexact->exact x)

  (define (type-error)
    (##fail-check-number 1 inexact->exact x))

  (define (range-error)
    (##raise-range-exception 1 inexact->exact x))

  (macro-number-dispatch x (type-error)
    x
    x
    x
    (if (macro-flonum-rational? x)
      (##flonum.inexact->exact x)
      (range-error))
    (let ((real (macro-cpxnum-real x))
          (imag (macro-cpxnum-imag x)))
      (if (and (macro-noncpxnum-rational? real)
               (macro-noncpxnum-rational? imag))
        (##make-rectangular (##inexact->exact real)
                            (##inexact->exact imag))
        (range-error)))))

(define-prim (inexact->exact x)
  (macro-force-vars (x)
    (##inexact->exact x)))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; number->string, string->number

(define-prim (##exact-int.number->string x rad force-sign?)

  (##define-macro (macro-make-block-size)
    (let* ((max-rad 16)
           (t (make-vector (+ max-rad 1) 0)))

      (define max-fixnum 536870911) ; OK to be conservative

      (define (block-size-for rad)
        (let loop ((i 0) (rad^i 1))
          (let ((new-rad^i (* rad^i rad)))
            (if (<= new-rad^i max-fixnum)
              (loop (+ i 1) new-rad^i)
              i))))

      (let loop ((i max-rad))
        (if (< 1 i)
          (begin
            (vector-set! t i (block-size-for i))
            (loop (- i 1)))))

      `',t))

  (define block-size (macro-make-block-size))

  (##define-macro (macro-make-rad^block-size)
    (let* ((max-rad 16)
           (t (make-vector (+ max-rad 1) 0)))

      (define max-fixnum 536870911) ; OK to be conservative

      (define (rad^block-size-for rad)
        (let loop ((i 0) (rad^i 1))
          (let ((new-rad^i (* rad^i rad)))
            (if (<= new-rad^i max-fixnum)
              (loop (+ i 1) new-rad^i)
              rad^i))))

      (let loop ((i max-rad))
        (if (< 1 i)
          (begin
            (vector-set! t i (rad^block-size-for i))
            (loop (- i 1)))))

      `',t))

  (define rad^block-size (macro-make-rad^block-size))

  (define (make-string-from-last-fixnum rad x len pos)
    (let loop ((x x) (len len) (pos pos))
      (if (##fixnum.= x 0)
        (##make-string len)
        (let* ((new-pos
                (##fixnum.+ pos 1))
               (s
                (loop (##fixnum.quotient x rad)
                      (##fixnum.+ len 1)
                      new-pos)))
          (##string-set!
           s
           (##fixnum.- (##string-length s) new-pos)
           (##string-ref ##digit-to-char-table
                         (##fixnum.- (##fixnum.remainder x rad))))
          s))))

  (define (convert-non-last-fixnum s rad x pos)
    (let loop ((x x)
               (size (##vector-ref block-size rad))
               (i (##fixnum.- (##string-length s) pos)))
      (if (##fixnum.< 0 size)
        (let ((new-i (##fixnum.- i 1)))
          (##string-set!
           s
           new-i
           (##string-ref ##digit-to-char-table
                         (##fixnum.remainder x rad)))
          (loop (##fixnum.quotient x rad)
                (##fixnum.- size 1)
                new-i)))))

  (define (make-string-from-fixnums rad lst len pos)
    (let loop ((lst lst) (pos pos))
      (let ((new-lst (##cdr lst)))
        (if (##null? new-lst)
          (make-string-from-last-fixnum
           rad
           (##fixnum.- (##car lst))
           (##fixnum.+ len pos)
           pos)
          (let* ((size
                  (##vector-ref block-size rad))
                 (new-pos
                  (##fixnum.+ pos size))
                 (s
                  (loop new-lst new-pos)))
            (convert-non-last-fixnum s rad (##car lst) pos)
            s)))))

  (define (uinteger->fixnums level sqs x lst)
    (cond ((and (##null? lst) (##eq? x 0))
           lst)
          ((##fixnum.= level 0)
           (##cons x lst))
          (else
           (let* ((qr (##exact-int.div x (##car sqs)))
                  (new-level (##fixnum.- level 1))
                  (new-sqs (##cdr sqs))
                  (q (##car qr))
                  (r (##cdr qr)))
             (uinteger->fixnums
              new-level
              new-sqs
              r
              (uinteger->fixnums new-level new-sqs q lst))))))

  (define (uinteger->string x rad len)
    (make-string-from-fixnums
     rad
     (let ((rad^size
            (##vector-ref rad^block-size rad))
           (x-length
            (##integer-length x)))
       (let loop ((level 0)
                  (sqs '())
                  (rad^size^2^level rad^size))
         (let ((new-level
                (##fixnum.+ level 1))
               (new-sqs
                (##cons rad^size^2^level sqs)))
           (if (##fixnum.< x-length
                           (##fixnum.-
                            (##fixnum.* (##integer-length rad^size^2^level) 2)
                            1))
             (uinteger->fixnums new-level new-sqs x '())
             (let ((new-rad^size^2^level
                    (##exact-int.square rad^size^2^level)))
               (if (##< x new-rad^size^2^level)
                 (uinteger->fixnums new-level new-sqs x '())
                 (loop new-level
                       new-sqs
                       new-rad^size^2^level)))))))
     len
     0))

  (if (##fixnum? x)

    (cond ((##fixnum.negative? x)
           (let ((s (make-string-from-last-fixnum rad x 1 0)))
             (##string-set! s 0 #\-)
             s))
          ((##fixnum.zero? x)
           (if force-sign?
             (##string #\+ #\0)
             (##string #\0)))
          (else
           (if force-sign?
             (let ((s (make-string-from-last-fixnum rad (##fixnum.- x) 1 0)))
               (##string-set! s 0 #\+)
               s)
             (make-string-from-last-fixnum rad (##fixnum.- x) 0 0))))

    (cond ((##bignum.negative? x)
           (let ((s (uinteger->string (##negate x) rad 1)))
             (##string-set! s 0 #\-)
             s))
          (else
           (if force-sign?
             (let ((s (uinteger->string x rad 1)))
               (##string-set! s 0 #\+)
               s)
             (uinteger->string x rad 0))))))

(define ##digit-to-char-table "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ")

(define-prim (##ratnum.number->string x rad force-sign?)
  (##string-append
   (##exact-int.number->string (macro-ratnum-numerator x) rad force-sign?)
   "/"
   (##exact-int.number->string (macro-ratnum-denominator x) rad #f)))

(##define-macro (macro-chez-fp-syntax) #f)

(##define-macro (macro-make-10^constants)
  (define n 326)
  (let ((v (make-vector n)))
    (let loop ((i 0) (x 1))
      (if (< i n)
        (begin
          (vector-set! v i x)
          (loop (+ i 1) (* x 10)))))
    `',v))

(define ##10^-constants
  (if (use-fast-bignum-algorithms)
    (macro-make-10^constants)
    #f))

(define-prim (##flonum.printout v sign-prefix)

  ; This algorithm is derived from the paper "Printing Floating-Point
  ; Numbers Quickly and Accurately" by Robert G. Burger and R. Kent Dybvig,
  ; SIGPLAN'96 Conference on Programming Language Design an Implementation.

  ; v is a flonum
  ; f is an exact integer (fixnum or bignum)
  ; e is an exact integer (fixnum only)

  (define (10^ n) ; 0 <= n < 326
    (if (use-fast-bignum-algorithms)
        (##vector-ref ##10^-constants n)
        (##expt 10 n)))

  (define (base-10-log x)
    (##define-macro (1/log10) `',(/ (log 10)))
    (##flonum.* (##flonum.log x) (1/log10)))

  (##define-macro (epsilon)
    1e-10)

  (define (scale r s m+ m- round? v)

    ; r is an exact integer (fixnum or bignum)
    ; s is an exact integer (fixnum or bignum)
    ; m+ is an exact integer (fixnum or bignum)
    ; m- is an exact integer (fixnum or bignum)
    ; round? is a boolean
    ; v is a flonum

    (let ((est
           (##flonum.->fixnum
            (##flonum.ceiling (##flonum.- (base-10-log v) (epsilon))))))
      (if (##fixnum.negative? est)
        (let ((factor (10^ (##fixnum.- est))))
          (fixup (##* r factor)
                 s
                 (##* m+ factor)
                 (##* m- factor)
                 est
                 round?))
        (let ((factor (10^ est)))
          (fixup r
                 (##* s factor)
                 m+
                 m-
                 est
                 round?)))))

  (define (fixup r s m+ m- k round?)
    (if (if round?
          (##not (##< (##+ r m+) s))
          (##< s (##+ r m+)))
      (##cons (##fixnum.+ k 1)
              (generate r
                        s
                        m+
                        m-
                        round?
                        0))
      (##cons k
              (generate (##* r 10)
                        s
                        (##* m+ 10)
                        (##* m- 10)
                        round?
                        0))))

  (define (generate r s m+ m- round? n)
    (let* ((dr (##exact-int.div r s))
           (d (##car dr))
           (r (##cdr dr))
           (tc (if round?
                 (##not (##< (##+ r m+) s))
                 (##< s (##+ r m+)))))
      (if (if round? (##not (##< m- r)) (##< r m-))
        (let* ((last-digit
                (if tc
                  (let ((r*2 (##arithmetic-shift r 1)))
                    (if (or (and (##fixnum.even? d)
                                 (##= r*2 s)) ; tie, round d to even
                            (##< r*2 s))
                      d
                      (##fixnum.+ d 1)))
                  d))
               (str
                (##make-string (##fixnum.+ n 1))))
          (##string-set! str n
            (##string-ref ##digit-to-char-table last-digit))
          str)
        (if tc
          (let ((str
                 (##make-string (##fixnum.+ n 1))))
            (##string-set! str n
              (##string-ref ##digit-to-char-table (##fixnum.+ d 1)))
            str)
          (let ((str
                 (generate (##* r 10)
                           s
                           (##* m+ 10)
                           (##* m- 10)
                           round?
                           (##fixnum.+ n 1))))
            (##string-set! str n
              (##string-ref ##digit-to-char-table d))
            str)))))

  (define (flonum->exponent-and-digits v)
    (let* ((x (##flonum.->exact-exponential-format v))
           (f (##vector-ref x 0))
           (e (##vector-ref x 1))
           (round? (##not (##odd? f))))
      (if (##fixnum.negative? e)
        (if (and (##not (##fixnum.= e (macro-flonum-e-min)))
                 (##= f (macro-flonum-+m-min)))
          (scale (##arithmetic-shift f 2)
                 (##arithmetic-shift 1 (##fixnum.- 2 e))
                 2
                 1
                 round?
                 v)
          (scale (##arithmetic-shift f 1)
                 (##arithmetic-shift 1 (##fixnum.- 1 e))
                 1
                 1
                 round?
                 v))
        (let ((2^e (##arithmetic-shift 1 e)))
          (if (##= f (macro-flonum-+m-min))
            (scale (##arithmetic-shift f (##fixnum.+ e 2))
                   4
                   (##arithmetic-shift 1 (##fixnum.+ e 1))
                   2^e
                   round?
                   v)
            (scale (##arithmetic-shift f (##fixnum.+ e 1))
                   2
                   2^e
                   2^e
                   round?
                   v))))))

  (let* ((x (flonum->exponent-and-digits v))
         (e (##car x))
         (d (##cdr x))            ; d = digits
         (n (##string-length d))) ; n = number of digits

    (cond ((and (##not (##fixnum.< e 0)) ; 0<=e<=10
                (##not (##fixnum.< 10 e)))

           (cond ((##fixnum.= e 0) ; e=0

                  ; Format 1: .DDD    (0.DDD in chez-fp-syntax)

                  (##string-append sign-prefix
                                   (if (macro-chez-fp-syntax) "0." ".")
                                   d))

                 ((##fixnum.< e n) ; e<n

                  ; Format 2: D.DDD up to DDD.D

                  (##string-append sign-prefix
                                   (##substring d 0 e)
                                   "."
                                   (##substring d e n)))

                 ((##fixnum.= e n) ; e=n

                  ; Format 3: DDD.    (DDD.0 in chez-fp-syntax)

                  (##string-append sign-prefix
                                   d
                                   (if (macro-chez-fp-syntax) ".0" ".")))

                 (else ; e>n

                  ; Format 4: DDD000000.    (DDD000000.0 in chez-fp-syntax)

                  (##string-append sign-prefix
                                   d
                                   (##make-string (##fixnum.- e n) #\0)
                                   (if (macro-chez-fp-syntax) ".0" ".")))))

          ((and (##not (##fixnum.< e -2)) ; -2<=e<=-1
                (##not (##fixnum.< -1 e)))

           ; Format 5: .0DDD or .00DDD    (0.0DDD or 0.00DDD in chez-fp-syntax)

           (##string-append sign-prefix
                            (if (macro-chez-fp-syntax) "0." ".")
                            (##make-string (##fixnum.- e) #\0)
                            d))

          (else

           ; Format 6: D.DDDeEEE
           ;
           ; This is the most general format.  We insert a period after
           ; the first digit (unless there is only one digit) and add
           ; an exponent.

           (##string-append sign-prefix
                            (##substring d 0 1)
                            (if (##fixnum.= n 1) "" ".")
                            (##substring d 1 n)
                            "e"
                            (##number->string (##fixnum.- e 1) 10))))))

(define-prim (##flonum.number->string x rad force-sign?)

  (define (non-neg-num->str x rad sign-prefix)
    (if (##flonum.zero? x)
      (##string-append sign-prefix (if (macro-chez-fp-syntax) "0.0" "0."))
      (##flonum.printout x sign-prefix)))

  (cond ((##flonum.nan? x)
         (##string-copy (if (macro-chez-fp-syntax) "+nan.0" "+nan.")))
        ((##flonum.negative? (##flonum.copysign (macro-inexact-+1) x))
         (let ((abs-x (##flonum.copysign x (macro-inexact-+1))))
           (cond ((##flonum.= abs-x (macro-inexact-+inf))
                  (##string-copy (if (macro-chez-fp-syntax) "-inf.0" "-inf.")))
                 (else
                  (non-neg-num->str abs-x rad "-")))))
        (else
         (cond ((##flonum.= x (macro-inexact-+inf))
                (##string-copy (if (macro-chez-fp-syntax) "+inf.0" "+inf.")))
               (force-sign?
                (non-neg-num->str x rad "+"))
               (else
                (non-neg-num->str x rad ""))))))

(define-prim (##cpxnum.number->string x rad force-sign?)
  (let* ((real
          (macro-cpxnum-real x))
         (real-str
          (if (##eq? real 0) "" (##number->string real rad force-sign?))))
    (let ((imag (macro-cpxnum-imag x)))
      (cond ((##eq? imag 1)
             (##string-append real-str "+i"))
            ((##eq? imag -1)
             (##string-append real-str "-i"))
            (else
             (##string-append real-str
                              (##number->string imag rad #t)
                              "i"))))))

(define-prim (##number->string x #!optional (rad 10) (force-sign? #f))
  (macro-number-dispatch x '()
    (##exact-int.number->string x rad force-sign?)
    (##exact-int.number->string x rad force-sign?)
    (##ratnum.number->string x rad force-sign?)
    (##flonum.number->string x rad force-sign?)
    (##cpxnum.number->string x rad force-sign?)))

(define-prim (number->string n #!optional (r (macro-absent-obj)))
  (macro-force-vars (n r)
    (let ((rad (if (##eq? r (macro-absent-obj)) 10 r)))
      (if (macro-exact-int? rad)
        (if (or (##eq? rad 2)
                (##eq? rad 8)
                (##eq? rad 10)
                (##eq? rad 16))
          (let ((result (##number->string n rad #f)))
            (if (##null? result)
              (##fail-check-number 1 number->string n r)
              result))
          (##raise-range-exception 2 number->string n r))
        (##fail-check-exact-integer 2 number->string n r)))))

(##define-macro (macro-make-char-to-digit-table)
  (let ((t (make-vector 128 99)))
    (vector-set! t (char->integer #\#) 0) ; #\# counts as 0
    (let loop1 ((i 9))
      (if (not (< i 0))
        (begin
          (vector-set! t (+ (char->integer #\0) i) i)
          (loop1 (- i 1)))))
    (let loop2 ((i 25))
      (if (not (< i 0))
        (begin
          (vector-set! t (+ (char->integer #\A) i) (+ i 10))
          (vector-set! t (+ (char->integer #\a) i) (+ i 10))
          (loop2 (- i 1)))))
    `',(list->u8vector (vector->list t))))

(define ##char-to-digit-table (macro-make-char-to-digit-table))

(define-prim (##string->number str #!optional (rad 10) (check-only? #f))

  ; The number grammar parsed by this procedure is:
  ;
  ; <num R E> : <prefix R E> <complex R E>
  ;
  ; <complex R E> : <real R E>
  ;               | <real R E> @ <real R E>
  ;               | <real R E> <sign> <ureal R> i
  ;               | <real R E> <sign-inf-nan R E> i
  ;               | <real R E> <sign> i
  ;               | <sign> <ureal R> i
  ;               | <sign-inf-nan R E> i
  ;               | <sign> i
  ;
  ; <real R E> : <ureal R>
  ;            | <sign> <ureal R>
  ;            | <sign-inf-nan R E>
  ;
  ; <sign-inf-nan R i> : +inf.
  ;                    | -inf.
  ;                    | +nan.
  ; <sign-inf-nan R empty> : <sign-inf-nan R i>
  ;
  ; <ureal R> : <uinteger R>
  ;           | <uinteger R> / <uinteger R>
  ;           | <decimal R>
  ;
  ; <decimal 10> : <uinteger 10> <suffix>
  ;              | . <digit 10>+ #* <suffix>
  ;              | <digit 10>+ . <digit 10>* #* <suffix>
  ;              | <digit 10>+ #+ . #* <suffix>
  ;
  ; <uinteger R> : <digit R>+ #*
  ;
  ; <prefix R E> : <radix R E> <exactness E>
  ;              | <exactness E> <radix R E>
  ;
  ; <suffix> : <empty>
  ;          | <exponent marker> <digit 10>+
  ;          | <exponent marker> <sign> <digit 10>+
  ;
  ; <exponent marker> : e | s | f | d | l
  ; <sign> : + | -
  ; <exactness empty> : <empty>
  ; <exactness i> : #i
  ; <exactness e> : #e
  ; <radix 2> : #b
  ; <radix 8> : #o
  ; <radix 10> : <empty> | #d
  ; <radix 16> : #x
  ; <digit 2> : 0 | 1
  ; <digit 8> : 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7
  ; <digit 10> : 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9
  ; <digit 16> : <digit 10> | a | b | c | d | e | f

  (##define-macro (macro-make-exact-10^n-table)

    (define max-exact-power-of-10 22) ; (floor (inexact->exact (/ (log (expt 2 (macro-flonum-m-bits-plus-1))) (log 5))))

    (let ((t (make-vector (+ max-exact-power-of-10 1))))

      (let loop ((i max-exact-power-of-10))
        (if (not (< i 0))
          (begin
            (vector-set! t i (exact->inexact (expt 10 i)))
            (loop (- i 1)))))

      `',(list->f64vector (vector->list t))))

  (define exact-10^n-table (macro-make-exact-10^n-table))

  (##define-macro (macro-make-block-size)
    (let* ((max-rad 16)
           (t (make-vector (+ max-rad 1) 0)))

      (define max-fixnum 536870911) ; OK to be conservative

      (define (block-size-for rad)
        (let loop ((i 0) (rad^i 1))
          (let ((new-rad^i (* rad^i rad)))
            (if (<= new-rad^i max-fixnum)
              (loop (+ i 1) new-rad^i)
              i))))

      (let loop ((i max-rad))
        (if (< 1 i)
          (begin
            (vector-set! t i (block-size-for i))
            (loop (- i 1)))))

      `',t))

  (define block-size (macro-make-block-size))

  (##define-macro (macro-make-rad^block-size)
    (let* ((max-rad 16)
           (t (make-vector (+ max-rad 1) 0)))

      (define max-fixnum 536870911) ; OK to be conservative

      (define (rad^block-size-for rad)
        (let loop ((i 0) (rad^i 1))
          (let ((new-rad^i (* rad^i rad)))
            (if (<= new-rad^i max-fixnum)
              (loop (+ i 1) new-rad^i)
              rad^i))))

      (let loop ((i max-rad))
        (if (< 1 i)
          (begin
            (vector-set! t i (rad^block-size-for i))
            (loop (- i 1)))))

      `',t))

  (define rad^block-size (macro-make-rad^block-size))

  (define (substring->uinteger-fixnum str rad i j)

    ; Simple case: result is known to fit in a fixnum.

    (let loop ((i i) (n 0))
      (if (##fixnum.< i j)
        (let ((c (##string-ref str i)))
          (if (##char<? c 128)
            (loop (##fixnum.+ i 1)
                  (##fixnum.+ (##fixnum.* n rad)
                              (##u8vector-ref ##char-to-digit-table c)))
            (loop (##fixnum.+ i 1)
                  (##fixnum.* n rad))))
        n)))

  (define (substring->uinteger-aux sqs width str rad i j)

    ; Divide-and-conquer algorithm (fast for large bignums if bignum
    ; multiplication is fast).

    (if (##null? sqs)
      (substring->uinteger-fixnum str rad i j)
      (let* ((new-sqs (##cdr sqs))
             (new-width (##fixnum.quotient width 2))
             (mid (##fixnum.- j new-width)))
        (if (##fixnum.< i mid)
          (let* ((a (substring->uinteger-aux new-sqs new-width str rad i mid))
                 (b (substring->uinteger-aux new-sqs new-width str rad mid j)))
            (##+ (##* a (##car sqs)) b))
          (substring->uinteger-aux new-sqs new-width str rad i j)))))

  (define (squares rad n)
    (let loop ((rad rad) (n n) (lst '()))
      (if (##fixnum.= n 1)
        (##cons rad lst)
        (loop (##exact-int.square rad)
              (##fixnum.- n 1)
              (##cons rad lst)))))

  (define (substring->uinteger str rad i j)

    ; Converts a substring into an unsigned integer.  Selects a fast
    ; conversion algorithm when result fits in a fixnum.

    (let ((len (##fixnum.- j i))
          (size (##vector-ref block-size rad)))
      (if (##fixnum.< size len)
        (let ((levels
               (##integer-length (##fixnum.quotient (##fixnum.- len 1) size))))
          (substring->uinteger-aux
           (squares (##vector-ref rad^block-size rad) levels)
           (##fixnum.arithmetic-shift-left size levels)
           str
           rad
           i
           j))
        (substring->uinteger-fixnum str rad i j))))

  (define (float-substring->uinteger str i j)

    ; Converts a substring containing the decimals of a floating-point
    ; number into an unsigned integer (any period is simply skipped).

    (let loop1 ((i i) (n 0))
      (if (##not (##fixnum.< i j))
        n
        (let ((c (##string-ref str i)))
          (if (##char=? c #\.)
            (loop1 (##fixnum.+ i 1) n)
            (let ((new-n
                   (##fixnum.+ (##fixnum.* n 10)
                               (if (##char<? c 128)
                                 (##u8vector-ref ##char-to-digit-table c)
                                 0))))
              (if (##fixnum.< new-n (macro-max-fixnum32-div-10))
                (loop1 (##fixnum.+ i 1) new-n)
                (let loop2 ((i i) (n n))
                  (if (##not (##fixnum.< i j))
                    n
                    (let ((c (##string-ref str i)))
                      (if (##char=? c #\.)
                        (loop2 (##fixnum.+ i 1) n)
                        (let ((new-n
                               (##+
                                (##* n 10)
                                (if (##char<? c 128)
                                  (##u8vector-ref ##char-to-digit-table c)
                                  0))))
                          (loop2 (##fixnum.+ i 1) new-n)))))))))))))

  (define (uinteger str rad i)
    (and (##fixnum.< i (##string-length str))
         (let ((c (##string-ref str i)))
           (and (##char<? c 128)
                (##not (##char=? c #\#))
                (##fixnum.< (##u8vector-ref ##char-to-digit-table c) rad)
                (digits-and-sharps str rad (##fixnum.+ i 1))))))

  (define (digits-and-sharps str rad i)
    (let loop ((i i))
      (if (##fixnum.< i (##string-length str))
        (let ((c (##string-ref str i)))
          (if (##char<? c 128)
            (if (##char=? c #\#)
              (sharps str (##fixnum.+ i 1))
              (if (##fixnum.< (##u8vector-ref ##char-to-digit-table c) rad)
                (loop (##fixnum.+ i 1))
                i))
            i))
        i)))

  (define (sharps str i)
    (let loop ((i i))
      (if (##fixnum.< i (##string-length str))
        (if (##char=? (##string-ref str i) #\#)
          (loop (##fixnum.+ i 1))
          i)
        i)))

  (define (suffix str i1)
    (if (##fixnum.< (##fixnum.+ i1 1) (##string-length str))
      (let ((c1 (##string-ref str i1)))
        (if (or (##char=? c1 #\e) (##char=? c1 #\E)
                (##char=? c1 #\s) (##char=? c1 #\S)
                (##char=? c1 #\f) (##char=? c1 #\F)
                (##char=? c1 #\d) (##char=? c1 #\D)
                (##char=? c1 #\l) (##char=? c1 #\L))
          (let ((c2 (##string-ref str (##fixnum.+ i1 1))))
            (let ((i2
                   (if (or (##char=? c2 #\+) (##char=? c2 #\-))
                     (uinteger str 10 (##fixnum.+ i1 2))
                     (uinteger str 10 (##fixnum.+ i1 1)))))
              (if (and i2
                       (##not (##char=? (##string-ref str (##fixnum.- i2 1))
                                        #\#)))
                i2
                i1)))
          i1))
      i1))

  (define (ureal str rad e i1)
    (let ((i2 (uinteger str rad i1)))
      (if i2
        (if (##fixnum.< i2 (##string-length str))
          (let ((c (##string-ref str i2)))
            (cond ((##char=? c #\/)
                   (let ((i3 (uinteger str rad (##fixnum.+ i2 1))))
                     (and i3
                          (let ((inexact-num?
                                 (or (##eq? e 'i)
                                     (and (##not e)
                                          (or (##char=? (##string-ref
                                                         str
                                                         (##fixnum.- i2 1))
                                                        #\#)
                                              (##char=? (##string-ref
                                                         str
                                                         (##fixnum.- i3 1))
                                                        #\#))))))
                            (if (and (##not inexact-num?)
                                     (##eq? (substring->uinteger
                                             str
                                             rad
                                             (##fixnum.+ i2 1)
                                             i3)
                                            0))
                              #f
                              (##vector i3 i2))))))
                  ((##fixnum.= rad 10)
                   (if (##char=? c #\.)
                     (let ((i3
                            (if (##char=? (##string-ref str (##fixnum.- i2 1))
                                          #\#)
                              (sharps str (##fixnum.+ i2 1))
                              (digits-and-sharps str 10 (##fixnum.+ i2 1)))))
                       (and i3
                            (let ((i4 (suffix str i3)))
                              (##vector i4 i3 i2))))
                     (let ((i3 (suffix str i2)))
                       (if (##fixnum.= i2 i3)
                         i2
                         (##vector i3 i2 i2)))))
                  (else
                   i2)))
          i2)
        (and (##fixnum.= rad 10)
             (##fixnum.< i1 (##string-length str))
             (##char=? (##string-ref str i1) #\.)
             (let ((i3 (uinteger str rad (##fixnum.+ i1 1))))
               (and i3
                    (let ((i4 (suffix str i3)))
                      (##vector i4 i3 i1))))))))

  (define (inf-nan str sign i e)
    (and (##not (##eq? e 'e))
         (if (##fixnum.< (##fixnum.+ i (if (macro-chez-fp-syntax) 4 3))
                         (##string-length str))
           (and (##char=? (##string-ref str (##fixnum.+ i 3)) #\.)
                (if (macro-chez-fp-syntax)
                  (##char=? (##string-ref str (##fixnum.+ i 4)) #\0)
                  #t)
                (or (and (let ((c (##string-ref str i)))
                           (or (##char=? c #\i) (##char=? c #\I)))
                         (let ((c (##string-ref str (##fixnum.+ i 1))))
                           (or (##char=? c #\n) (##char=? c #\N)))
                         (let ((c (##string-ref str (##fixnum.+ i 2))))
                           (or (##char=? c #\f) (##char=? c #\F))))
                    (and (##not (##char=? sign #\-))
                         (let ((c (##string-ref str i)))
                           (or (##char=? c #\n) (##char=? c #\N)))
                         (let ((c (##string-ref str (##fixnum.+ i 1))))
                           (or (##char=? c #\a) (##char=? c #\A)))
                         (let ((c (##string-ref str (##fixnum.+ i 2))))
                           (or (##char=? c #\n) (##char=? c #\N)))))
                (##vector (##fixnum.+ i (if (macro-chez-fp-syntax) 5 4))))
           #f)))

  (define (make-rec x y)
    (##make-rectangular x y))

  (define (make-pol x y e)
    (let ((n (##make-polar x y)))
      (if (##eq? e 'e)
        (##inexact->exact n)
        n)))

  (define (make-inexact-real sign uinteger exponent)
    (let ((n
           (if (and (##fixnum? uinteger)
                    (##flonum.<-fixnum-exact? uinteger)
                    (##fixnum? exponent)
                    (##fixnum.< (##fixnum.- exponent)
                                (##f64vector-length exact-10^n-table))
                    (##fixnum.< exponent
                                (##f64vector-length exact-10^n-table)))
             (if (##fixnum.< exponent 0)
               (##flonum./ (##flonum.<-fixnum uinteger)
                           (##f64vector-ref exact-10^n-table
                                            (##fixnum.- exponent)))
               (##flonum.* (##flonum.<-fixnum uinteger)
                           (##f64vector-ref exact-10^n-table
                                            exponent)))
             (##exact->inexact
              (##* uinteger (##expt 10 exponent))))))
      (if (##char=? sign #\-)
        (##flonum.copysign n (macro-inexact--1))
        n)))

  (define (get-zero e)
    (if (##eq? e 'i)
      (macro-inexact-+0)
      0))

  (define (get-one sign e)
    (if (##eq? e 'i)
      (if (##char=? sign #\-) (macro-inexact--1) (macro-inexact-+1))
      (if (##char=? sign #\-) -1 1)))

  (define (get-real start sign str rad e i)
    (if (##fixnum? i)
      (let* ((abs-n
              (substring->uinteger str rad start i))
             (n
              (if (##char=? sign #\-)
                (##negate abs-n)
                abs-n)))
        (if (or (##eq? e 'i)
                (and (##not e)
                     (##char=? (##string-ref str (##fixnum.- i 1)) #\#)))
          (##exact->inexact n)
          n))
      (let ((j (##vector-ref i 0))
            (len (##vector-length i)))
        (cond ((##fixnum.= len 3) ; xxx.yyyEzzz
               (let* ((after-frac-part
                       (##vector-ref i 1))
                      (unadjusted-exponent
                       (if (##fixnum.= after-frac-part j) ; no exponent part?
                         0
                         (let* ((c
                                 (##string-ref
                                  str
                                  (##fixnum.+ after-frac-part 1)))
                                (n
                                 (substring->uinteger
                                  str
                                  10
                                  (if (or (##char=? c #\+) (##char=? c #\-))
                                    (##fixnum.+ after-frac-part 2)
                                    (##fixnum.+ after-frac-part 1))
                                  j)))
                           (if (##char=? c #\-)
                             (##negate n)
                             n))))
                      (c
                       (##string-ref str start))
                      (uinteger
                       (float-substring->uinteger str start after-frac-part))
                      (decimals-after-point
                       (##fixnum.-
                        (##fixnum.- after-frac-part (##vector-ref i 2))
                        1))
                      (exponent
                       (if (##fixnum.< 0 decimals-after-point)
                         (if (and (##fixnum? unadjusted-exponent)
                                  (##fixnum.< (##fixnum.- unadjusted-exponent
                                                          decimals-after-point)
                                              unadjusted-exponent))
                           (##fixnum.- unadjusted-exponent
                                       decimals-after-point)
                           (##- unadjusted-exponent
                                decimals-after-point))
                         unadjusted-exponent)))
                 (if (##eq? e 'e)
                   (##*
                    (if (##char=? sign #\-)
                      (##negate uinteger)
                      uinteger)
                    (##expt 10 exponent))
                   (make-inexact-real sign uinteger exponent))))
              ((##fixnum.= len 2) ; xxx/yyy
               (let* ((after-num
                       (##vector-ref i 1))
                      (inexact-num?
                       (or (##eq? e 'i)
                           (and (##not e)
                                (or (##char=? (##string-ref
                                               str
                                               (##fixnum.- after-num 1))
                                              #\#)
                                    (##char=? (##string-ref
                                               str
                                               (##fixnum.- j 1))
                                              #\#)))))
                      (abs-num
                       (substring->uinteger str rad start after-num))
                      (den
                       (substring->uinteger str
                                            rad
                                            (##fixnum.+ after-num 1)
                                            j)))

                 (define (num-div-den)
                   (##/ (if (##char=? sign #\-)
                          (##negate abs-num)
                          abs-num)
                        den))

                 (if inexact-num?
                   (if (##eq? den 0)
                     (let ((n
                            (if (##eq? abs-num 0)
                              (macro-inexact-+nan)
                              (macro-inexact-+inf))))
                       (if (##char=? sign #\-)
                         (##flonum.copysign n (macro-inexact--1))
                         n))
                     (##inexact->exact (num-div-den)))
                   (num-div-den))))
              (else ; (##fixnum.= len 1) ; inf or nan
               (let* ((c
                       (##string-ref str start))
                      (n
                       (if (or (##char=? c #\i) (##char=? c #\I))
                         (macro-inexact-+inf)
                         (macro-inexact-+nan))))
                 (if (##char=? sign #\-)
                   (##flonum.copysign n (macro-inexact--1))
                   n)))))))

  (define (i-end str i)
    (and (##fixnum.= (##fixnum.+ i 1) (##string-length str))
         (let ((c (##string-ref str i)))
           (or (##char=? c #\i) (##char=? c #\I)))))

  (define (complex start sign str rad e i)
    (let ((j (if (##fixnum? i) i (##vector-ref i 0))))
      (let ((c (##string-ref str j)))
        (cond ((##char=? c #\@)
               (let ((j+1 (##fixnum.+ j 1)))
                 (if (##fixnum.< j+1 (##string-length str))
                   (let* ((sign2
                           (##string-ref str j+1))
                          (start2
                           (if (or (##char=? sign2 #\+) (##char=? sign2 #\-))
                             (##fixnum.+ j+1 1)
                             j+1))
                          (k
                           (or (ureal str rad e start2)
                               (and (##fixnum.< j+1 start2)
                                    (inf-nan str sign2 start2 e)))))
                     (and k
                          (let ((l (if (##fixnum? k) k (##vector-ref k 0))))
                            (and (##fixnum.= l (##string-length str))
                                 (or check-only?
                                     (make-pol
                                      (get-real start sign str rad e i)
                                      (get-real start2 sign2 str rad e k)
                                      e))))))
                   #f)))
              ((or (##char=? c #\+) (##char=? c #\-))
               (let* ((start2
                       (##fixnum.+ j 1))
                      (k
                       (or (ureal str rad e start2)
                           (inf-nan str c start2 e))))
                 (if (##not k)
                   (if (i-end str start2)
                     (or check-only?
                         (make-rec
                          (get-real start sign str rad e i)
                          (get-one c e)))
                     #f)
                   (let ((l (if (##fixnum? k) k (##vector-ref k 0))))
                     (and (i-end str l)
                          (or check-only?
                              (make-rec
                               (get-real start sign str rad e i)
                               (get-real start2 c str rad e k))))))))
              (else
               #f)))))

  (define (after-prefix start str rad e)

    ; invariant: start = 0, 2 or 4, (string-length str) > start

    (let ((c (##string-ref str start)))
      (if (or (##char=? c #\+) (##char=? c #\-))
        (let ((i (or (ureal str rad e (##fixnum.+ start 1))
                     (inf-nan str c (##fixnum.+ start 1) e))))
          (if (##not i)
            (if (i-end str (##fixnum.+ start 1))
              (or check-only?
                  (make-rec
                   (get-zero e)
                   (get-one c e)))
              #f)
            (let ((j (if (##fixnum? i) i (##vector-ref i 0))))
              (cond ((##fixnum.= j (##string-length str))
                     (or check-only?
                         (get-real (##fixnum.+ start 1) c str rad e i)))
                    ((i-end str j)
                     (or check-only?
                         (make-rec
                          (get-zero e)
                          (get-real (##fixnum.+ start 1) c str rad e i))))
                    (else
                     (complex (##fixnum.+ start 1) c str rad e i))))))
        (let ((i (ureal str rad e start)))
          (if (##not i)
            #f
            (let ((j (if (##fixnum? i) i (##vector-ref i 0))))
              (cond ((##fixnum.= j (##string-length str))
                     (or check-only?
                         (get-real start #\+ str rad e i)))
                    (else
                     (complex start #\+ str rad e i)))))))))

  (define (radix-prefix c)
    (cond ((or (##char=? c #\b) (##char=? c #\B))  2)
          ((or (##char=? c #\o) (##char=? c #\O))  8)
          ((or (##char=? c #\d) (##char=? c #\D)) 10)
          ((or (##char=? c #\x) (##char=? c #\X)) 16)
          (else                                   #f)))

  (define (exactness-prefix c)
    (cond ((or (##char=? c #\i) (##char=? c #\I)) 'i)
          ((or (##char=? c #\e) (##char=? c #\E)) 'e)
          (else                                   #f)))

  (cond ((##fixnum.< 2 (##string-length str)) ; >= 3 chars
         (if (##char=? (##string-ref str 0) #\#)
           (let ((rad1 (radix-prefix (##string-ref str 1))))
             (if rad1
               (if (and (##fixnum.< 4 (##string-length str)) ; >= 5 chars
                        (##char=? (##string-ref str 2) #\#))
                 (let ((e1 (exactness-prefix (##string-ref str 3))))
                   (if e1
                     (after-prefix 4 str rad1 e1)
                     #f))
                 (after-prefix 2 str rad1 #f))
               (let ((e2 (exactness-prefix (##string-ref str 1))))
                 (if e2
                   (if (and (##fixnum.< 4 (##string-length str)) ; >= 5 chars
                            (##char=? (##string-ref str 2) #\#))
                     (let ((rad2 (radix-prefix (##string-ref str 3))))
                       (if rad2
                         (after-prefix 4 str rad2 e2)
                         #f))
                     (after-prefix 2 str rad e2))
                   #f))))
           (after-prefix 0 str rad #f)))
        ((##fixnum.< 0 (##string-length str)) ; >= 1 char
         (after-prefix 0 str rad #f))
        (else
         #f)))

(define-prim (string->number str #!optional (r (macro-absent-obj)))
  (macro-force-vars (str r)
    (macro-check-string str 1 (string->number str r)
      (let ((rad (if (##eq? r (macro-absent-obj)) 10 r)))
        (if (macro-exact-int? rad)
          (if (or (##eq? rad 2)
                  (##eq? rad 8)
                  (##eq? rad 10)
                  (##eq? rad 16))
            (##string->number str rad #f)
            (##raise-range-exception 2 string->number str r))
          (##fail-check-exact-integer 2 string->number str r))))))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Bitwise operations.

(define-prim (##bitwise-ior x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (define (bignum-bitwise-ior x x-length y y-length)
    (if (##bignum.negative? x)
      (let ((result (##bignum.make x-length x #f)))
        (##declare (not interrupts-enabled))
        (let loop1 ((i 0))
          (if (##fixnum.< i x-length)
            (begin
              (##bignum.adigit-bitwise-ior! result i y i)
              (loop1 (##fixnum.+ i 1)))
            (##bignum.normalize! result))))
      (let ((result (##bignum.make y-length y #f)))
        (##declare (not interrupts-enabled))
        (let loop2 ((i 0))
          (if (##fixnum.< i x-length)
            (begin
              (##bignum.adigit-bitwise-ior! result i x i)
              (loop2 (##fixnum.+ i 1)))
            (##bignum.normalize! result))))))

  (cond ((##fixnum? x)
         (cond ((##fixnum? y)
                (##fixnum.bitwise-ior x y))
               ((##bignum? y)
                (let* ((x-bignum (##bignum.<-fixnum x))
                       (x-length (##bignum.adigit-length x-bignum))
                       (y-length (##bignum.adigit-length y)))
                  (bignum-bitwise-ior x-bignum x-length y y-length)))
               (else
                (type-error-on-y))))
        ((##bignum? x)
         (let ((x-length (##bignum.adigit-length x)))
           (cond ((##fixnum? y)
                  (let* ((y-bignum (##bignum.<-fixnum y))
                         (y-length (##bignum.adigit-length y-bignum)))
                    (bignum-bitwise-ior y-bignum y-length x x-length)))
                 ((##bignum? y)
                  (let ((y-length (##bignum.adigit-length y)))
                    (if (##fixnum.< x-length y-length)
                      (bignum-bitwise-ior x x-length y y-length)
                      (bignum-bitwise-ior y y-length x x-length))))
                 (else
                  (type-error-on-y)))))
        (else
         (type-error-on-x))))

(define-prim-nary (bitwise-ior x y)
  0
  (if (macro-exact-int? x) x '(1))
  (##bitwise-ior x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-exact-integer))

(define-prim (##bitwise-xor x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (define (bignum-bitwise-xor x x-length y y-length)
    (let ((result (##bignum.make y-length y #f)))
      (##declare (not interrupts-enabled))
      (let loop1 ((i 0))
        (if (##fixnum.< i x-length)
          (begin
            (##bignum.adigit-bitwise-xor! result i x i)
            (loop1 (##fixnum.+ i 1)))
          (if (##bignum.negative? x)
            (let loop2 ((i i))
              (if (##fixnum.< i y-length)
                (begin
                  (##bignum.adigit-bitwise-not! result i)
                  (loop2 (##fixnum.+ i 1)))
                (##bignum.normalize! result)))
            (##bignum.normalize! result))))))

  (cond ((##fixnum? x)
         (cond ((##fixnum? y)
                (##fixnum.bitwise-xor x y))
               ((##bignum? y)
                (let* ((x-bignum (##bignum.<-fixnum x))
                       (x-length (##bignum.adigit-length x-bignum))
                       (y-length (##bignum.adigit-length y)))
                  (bignum-bitwise-xor x-bignum x-length y y-length)))
               (else
                (type-error-on-y))))
        ((##bignum? x)
         (let ((x-length (##bignum.adigit-length x)))
           (cond ((##fixnum? y)
                  (let* ((y-bignum (##bignum.<-fixnum y))
                         (y-length (##bignum.adigit-length y-bignum)))
                    (bignum-bitwise-xor y-bignum y-length x x-length)))
                 ((##bignum? y)
                  (let ((y-length (##bignum.adigit-length y)))
                    (if (##fixnum.< x-length y-length)
                      (bignum-bitwise-xor x x-length y y-length)
                      (bignum-bitwise-xor y y-length x x-length))))
                 (else
                  (type-error-on-y)))))
        (else
         (type-error-on-x))))

(define-prim-nary (bitwise-xor x y)
  0
  (if (macro-exact-int? x) x '(1))
  (##bitwise-xor x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-exact-integer))

(define-prim (##bitwise-and x y)

  (##define-macro (type-error-on-x) `'(1))
  (##define-macro (type-error-on-y) `'(2))

  (define (bignum-bitwise-and x x-length y y-length)
    (if (##bignum.negative? x)
      (let ((result (##bignum.make y-length y #f)))
        (##declare (not interrupts-enabled))
        (let loop1 ((i 0))
          (if (##fixnum.< i x-length)
            (begin
              (##bignum.adigit-bitwise-and! result i x i)
              (loop1 (##fixnum.+ i 1)))
            (##bignum.normalize! result))))
      (let ((result (##bignum.make x-length x #f)))
        (##declare (not interrupts-enabled))
        (let loop2 ((i 0))
          (if (##fixnum.< i x-length)
            (begin
              (##bignum.adigit-bitwise-and! result i y i)
              (loop2 (##fixnum.+ i 1)))
            (##bignum.normalize! result))))))

  (cond ((##fixnum? x)
         (cond ((##fixnum? y)
                (##fixnum.bitwise-and x y))
               ((##bignum? y)
                (let* ((x-bignum (##bignum.<-fixnum x))
                       (x-length (##bignum.adigit-length x-bignum))
                       (y-length (##bignum.adigit-length y)))
                  (bignum-bitwise-and x-bignum x-length y y-length)))
               (else
                (type-error-on-y))))
        ((##bignum? x)
         (let ((x-length (##bignum.adigit-length x)))
           (cond ((##fixnum? y)
                  (let* ((y-bignum (##bignum.<-fixnum y))
                         (y-length (##bignum.adigit-length y-bignum)))
                    (bignum-bitwise-and y-bignum y-length x x-length)))
                 ((##bignum? y)
                  (let ((y-length (##bignum.adigit-length y)))
                    (if (##fixnum.< x-length y-length)
                      (bignum-bitwise-and x x-length y y-length)
                      (bignum-bitwise-and y y-length x x-length))))
                 (else
                  (type-error-on-y)))))
        (else
         (type-error-on-x))))

(define-prim-nary (bitwise-and x y)
  -1
  (if (macro-exact-int? x) x '(1))
  (##bitwise-and x y)
  macro-force-vars
  macro-no-check
  (##pair? ##fail-check-exact-integer))

(define-prim (##bitwise-not x)

  (define (type-error)
    (##fail-check-exact-integer 1 bitwise-not x))

  (cond ((##fixnum? x)
         (##fixnum.bitwise-not x))
        ((##bignum? x)
         (##bignum.make (##bignum.adigit-length x) x #t))
        (else
         (type-error))))

(define-prim (bitwise-not x)
  (macro-force-vars (x)
    (##bitwise-not x)))

(define-prim (##arithmetic-shift x y)

  (define (type-error-on-x)
    (##fail-check-exact-integer 1 arithmetic-shift x y))

  (define (type-error-on-y)
    (##fail-check-exact-integer 2 arithmetic-shift x y))

  (define (overflow)
    (##raise-heap-overflow-exception)
    (##arithmetic-shift x y))

  (define (general-fixnum-fixnum-case)
    (##bignum.arithmetic-shift (##bignum.<-fixnum x) y))

  (cond ((##fixnum? x)
         (cond ((##fixnum? y)
                (cond ((##fixnum.zero? y)
                       x)
                      ((##fixnum.negative? y)                  ;; right shift
                       (if (##fixnum.< (##fixnum.- ##fixnum-width) y)
                           (##fixnum.arithmetic-shift-right x (##fixnum.- y))
                           (if (##fixnum.negative? x)
                               -1
                               0)))
                      (else                                    ;; left shift
                       (if (##fixnum.< y ##fixnum-width)
                         (let ((result (##fixnum.arithmetic-shift-left x y)))
                           (if (##fixnum.=
                                (##fixnum.arithmetic-shift-right result y)
                                x)
                             result
                             (general-fixnum-fixnum-case)))
                         (general-fixnum-fixnum-case)))))
               ((##bignum? y)
                (cond ((##fixnum.zero? x)
                       0)
                      ((##bignum.negative? y)
                       (if (##fixnum.negative? x)
                         -1
                         0))
                      (else
                       (overflow))))
               (else
                (type-error-on-y))))
        ((##bignum? x)
         (cond ((##eq? y 0)
                x)
               ((##fixnum? y)
                (##bignum.arithmetic-shift x y))
               ((##bignum? y)
                (cond ((##bignum.negative? y)
                       (if (##bignum.negative? x)
                         -1
                         0))
                      (else
                       (overflow))))
               (else
                (type-error-on-y))))
        (else
         (type-error-on-x))))

(define-prim (arithmetic-shift x y)
  (macro-force-vars (x y)
    (##arithmetic-shift x y)))

(define-prim (##bit-count x)

  (define (type-error)
    (##fail-check-exact-integer 1 bit-count x))

  (define (at-most-6 x)
    (##u8vector-ref
     '#u8(0 1 1 2 1 2 2 3 1 2 2 3 2 3 3 4 1 2 2 3 2 3 3 4 2 3 3 4 3 4 4 5
          1 2 2 3 2 3 3 4 2 3 3 4 3 4 4 5 2 3 3 4 3 4 4 5 3 4 4 5 4 5 5 6)
     x))

  (define (count x n)
    (if (##fixnum.< x 64)
      (##fixnum.+ n (at-most-6 x))
      (count (##fixnum.arithmetic-shift-right x 6)
             (##fixnum.+ n (at-most-6 (##fixnum.bitwise-and x 63))))))

  (cond ((##fixnum? x)
         (if (##fixnum.negative? x)
           (count (##fixnum.bitwise-not x) 0)
           (count x 0)))
        ((##bignum? x)
         (let ((x-length (##bignum.mdigit-length x)))
           (let loop ((i (##fixnum.- x-length 1))
                      (n 0))
             (if (##fixnum.< i 0)
               (if (##bignum.negative? x)
                 (##fixnum.- (##fixnum.* x-length ##bignum.mdigit-width) n)
                 n)
               (loop (##fixnum.- i 1)
                     (count (##bignum.mdigit-ref x i) n))))))
        (else
         (type-error))))

(define-prim (bit-count x)
  (macro-force-vars (x)
    (##bit-count x)))

(define-prim (##fixnum.integer-length x n)

  (define (length x n)
    (if (##fixnum.< x 65536)
      (at-most-16 x n)
      (length (##fixnum.arithmetic-shift-right x 16) (##fixnum.+ n 16))))

  (define (at-most-16 x n)
    (if (##fixnum.< x 256)
      (at-most-8 x n)
      (at-most-8 (##fixnum.arithmetic-shift-right x 8) (##fixnum.+ n 8))))

  (define (at-most-8 x n)
    (if (##fixnum.< x 16)
      (at-most-4 x n)
      (at-most-4 (##fixnum.arithmetic-shift-right x 4) (##fixnum.+ n 4))))

  (define (at-most-4 x n)
    (##fixnum.+ (##u8vector-ref '#u8(0 1 2 2 3 3 3 3 4 4 4 4 4 4 4 4) x) n))

  (if (##fixnum.< x 0)
    (length (##fixnum.- -1 x) n)
    (length x n)))

(define-prim (##integer-length x)

  (define (type-error)
    (##fail-check-exact-integer 1 integer-length x))

  (cond ((##fixnum? x)
         (##fixnum.integer-length x 0))
        ((##bignum? x)
         (let ((x-length (##bignum.mdigit-length x)))
           (if (##bignum.negative? x)
             (let loop1 ((i (##fixnum.- x-length 1)))
               (let ((mdigit (##bignum.mdigit-ref x i)))
                 (if (##fixnum.= mdigit ##bignum.mdigit-base-minus-1)
                   (loop1 (##fixnum.- i 1))
                   (##fixnum.integer-length
                    (##fixnum.- ##bignum.mdigit-base-minus-1 mdigit)
                    (##fixnum.* i ##bignum.mdigit-width)))))
             (let loop2 ((i (##fixnum.- x-length 1)))
               (let ((mdigit (##bignum.mdigit-ref x i)))
                 (if (##fixnum.= mdigit 0)
                   (loop2 (##fixnum.- i 1))
                   (##fixnum.integer-length
                    mdigit
                    (##fixnum.* i ##bignum.mdigit-width))))))))
        (else
         (type-error))))

(define-prim (integer-length x)
  (macro-force-vars (x)
    (##integer-length x)))

(define-prim (##bitwise-merge x y z)
  (cond ((##not (macro-exact-int? x))
         (##fail-check-exact-integer 1 bitwise-merge x y z))
        ((##not (macro-exact-int? y))
         (##fail-check-exact-integer 2 bitwise-merge x y z))
        ((##not (macro-exact-int? z))
         (##fail-check-exact-integer 3 bitwise-merge x y z))
        (else
         (##bitwise-ior (##bitwise-and (##bitwise-not x) y)
                        (##bitwise-and x z)))))

(define-prim (bitwise-merge x y z)
  (macro-force-vars (x y z)
    (##bitwise-merge x y z)))

(define-prim (##bit-set? x y)

  (define (type-error-on-x)
    (##fail-check-exact-integer 1 bit-set? x y))

  (define (type-error-on-y)
    (##fail-check-exact-integer 2 bit-set? x y))

  (define (range-error)
    (##raise-range-exception 1 bit-set? x y))

  (cond ((##fixnum? x)
         (cond ((##fixnum? y)
                (if (##fixnum.negative? x)
                  (range-error)
                  (if (##fixnum.< x ##fixnum-width)
                    (##fixnum.odd? (##fixnum.arithmetic-shift-right y x))
                    (##fixnum.negative? y))))
               ((##bignum? y)
                (if (##fixnum.negative? x)
                  (range-error)
                  (let ((i (##fixnum.quotient x ##bignum.mdigit-width)))
                    (if (##fixnum.< i (##bignum.mdigit-length y))
                      (##fixnum.odd?
                       (##fixnum.arithmetic-shift-right
                        (##bignum.mdigit-ref y i)
                        (##fixnum.modulo x ##bignum.mdigit-width)))
                      (##bignum.negative? y)))))
               (else
                (type-error-on-y))))
        ((##bignum? x)
         (cond ((##fixnum? y)
                (if (##bignum.negative? x)
                  (range-error)
                  (##fixnum.negative? y)))
               ((##bignum? y)
                (if (##bignum.negative? x)
                  (range-error)
                  (##bignum.negative? y)))
               (else
                (type-error-on-y))))
        (else
         (type-error-on-x))))

(define-prim (bit-set? x y)
  (macro-force-vars (x y)
    (##bit-set? x y)))

(define-prim (##any-bits-set? x y)
  (cond ((##not (macro-exact-int? x))
         (##fail-check-exact-integer 1 any-bits-set? x y))
        ((##not (macro-exact-int? y))
         (##fail-check-exact-integer 2 any-bits-set? x y))
        (else
         (##not (##eq? (##bitwise-and x y) 0)))))

(define-prim (any-bits-set? x y)
  (macro-force-vars (x y)
    (##any-bits-set? x y)))

(define-prim (##all-bits-set? x y)
  (cond ((##not (macro-exact-int? x))
         (##fail-check-exact-integer 1 all-bits-set? x y))
        ((##not (macro-exact-int? y))
         (##fail-check-exact-integer 2 all-bits-set? x y))
        (else
         (##= x (##bitwise-and x y)))))

(define-prim (all-bits-set? x y)
  (macro-force-vars (x y)
    (##all-bits-set? x y)))

(define-prim (##first-set-bit x)

  (define (type-error)
    (##fail-check-exact-integer 1 first-set-bit x))

  (define (at-most-5 x)
    (##u8vector-ref
     '#u8(0 0 1 0 2 0 1 0 3 0 1 0 2 0 1 0 4 0 1 0 2 0 1 0 3 0 1 0 2 0 1 0
          5 0 1 0 2 0 1 0 3 0 1 0 2 0 1 0 4 0 1 0 2 0 1 0 3 0 1 0 2 0 1 0)
     x))

  (define (first x n)
    (let ((x-mod-64 (##fixnum.bitwise-and x 63)))
      (if (##fixnum.= x-mod-64 0)
        (first (##fixnum.arithmetic-shift-right x 6)
               (##fixnum.+ n 6))
        (##fixnum.+ n (at-most-5 x-mod-64)))))

  (cond ((##fixnum? x)
         (if (##fixnum.= x 0)
           #f
           (first x 0)))
        ((##bignum? x)
         (let ((x-length (##bignum.mdigit-length x)))
           (let loop ((i 0))
             (let ((mdigit (##bignum.mdigit-ref x i)))
               (if (##fixnum.= mdigit 0)
                 (loop (##fixnum.+ i 1))
                 (first mdigit (##fixnum.* i ##bignum.mdigit-width)))))))
        (else
         (type-error))))

(define-prim (first-set-bit x)
  (macro-force-vars (x)
    (##first-set-bit x)))

(define ##extract-bit-field-fixnum-limit
  (##fixnum.- ##fixnum-width 1))

(define (##extract-bit-field size position n)

  ;; size and position must be nonnegative

  (define (fixup-top-word result size)
    (##declare (not interrupts-enabled))
    (let ((size-words (##fixnum.quotient  size ##bignum.mdigit-width))
          (size-bits  (##fixnum.remainder size ##bignum.mdigit-width)))
      (let loop ((i (##fixnum.- (##bignum.mdigit-length result) 1)))
        (cond ((##fixnum.< size-words i)
               (##bignum.mdigit-set! result i 0)
               (loop (##fixnum.- i 1)))
              ((##eq? size-words i)
               (##bignum.mdigit-set!
                result i
                (##fixnum.bitwise-and
                 (##bignum.mdigit-ref result i)
                 (##fixnum.bitwise-not (##fixnum.arithmetic-shift-left -1 size-bits))))
               (##bignum.normalize! result))
              (else
               (##bignum.normalize! result))))))

  (cond ((##bignum? size)
         (if (##negative? n)
             (##bignum.make ##max-fixnum #f #f) ;; generates heap overflow
             (##arithmetic-shift n (##- position))))
        ((##bignum? position)
         (if (##negative? n)
             (##extract-bit-field size 0 -1)
             0))
        ((and (##fixnum? n)
              (##fixnum.< size ##extract-bit-field-fixnum-limit))
         (##fixnum.bitwise-and (##fixnum.arithmetic-shift-right
                                n
                                (##fixnum.min position ##extract-bit-field-fixnum-limit))
                               (##fixnum.bitwise-not (##fixnum.arithmetic-shift-left -1 size))))
        (else
         (let* ((n (if (##fixnum? n)
                       (##bignum.<-fixnum n)
                       n))
                (n-length (##bignum.adigit-length n))
                (n-negative? (##bignum.negative? n))
                (result-bit-size
                 (if n-negative?
                     size
                     (##fixnum.min
                      (##fixnum.- (##fixnum.* ##bignum.adigit-width
                                              n-length)
                                  position
                                  1)      ;; the top bit of a nonnegative bignum is always 0
                      size))))
           (if (##fixnum.<= result-bit-size 0)
               0
               (let* ((result-word-size
                       (##fixnum.+ (##fixnum.quotient result-bit-size
                                                      ##bignum.adigit-width)
                                   1))
                      (result (if (##eq? position 0)
                                  (##bignum.make result-word-size n #f)             ;; copy lowest result-word-size words of n to result
                                  (##bignum.make result-word-size #f n-negative?)))
                      (word-shift (##fixnum.quotient position ##bignum.adigit-width))
                      (bit-shift (##fixnum.remainder position ##bignum.adigit-width))
                      (divider (##fixnum.- ##bignum.adigit-width bit-shift))
                      )
                 (cond ((##eq? position 0)
                        (fixup-top-word result size))
                       ((##eq? bit-shift 0)
                        (let ((word-limit (##fixnum.min (##fixnum.+ word-shift result-word-size)
                                                        n-length)))
                          (##declare (not interrupts-enabled))
                          (let loop ((i 0)
                                     (j word-shift))
                            (if (##fixnum.< j word-limit)
                                (begin
                                  (##bignum.adigit-copy! result i n j)
                                  (loop (##fixnum.+ i 1)
                                        (##fixnum.+ j 1)))
                                (fixup-top-word result size)))))
                       (else
                        (let ((left-fill (if n-negative?
                                             ##bignum.adigit-ones
                                             ##bignum.adigit-zeros))
                              (word-limit (##fixnum.- (##fixnum.min (##fixnum.+ word-shift result-word-size)
                                                                    n-length)
                                                      1)))
                          (##declare (not interrupts-enabled))
                          (let loop ((i 0)
                                     (j word-shift))
                            (cond ((##fixnum.< j word-limit)
                                   (##bignum.adigit-cat! result i
                                                         n (##fixnum.+ j 1)
                                                         n j
                                                         divider)
                                   (loop (##fixnum.+ i 1)
                                         (##fixnum.+ j 1)))
                                  ((##fixnum.< j (##fixnum.- n-length 1))
                                   (##bignum.adigit-cat! result i
                                                         n (##fixnum.+ j 1)
                                                         n j
                                                         divider)
                                   (fixup-top-word result size))
                                  ((##fixnum.= j (##fixnum.- n-length 1))
                                   (##bignum.adigit-cat! result i
                                                         left-fill 0
                                                         n j
                                                         divider)
                                   (fixup-top-word result size))
                                  (else
                                   (fixup-top-word result size)))))))))))))

(define-prim (extract-bit-field size position n)
  (macro-force-vars (size position n)
    (cond ((##not (macro-exact-int? size))
         (##fail-check-exact-integer 1 extract-bit-field size position n))
        ((##not (macro-exact-int? position))
         (##fail-check-exact-integer 2 extract-bit-field size position n))
        ((##not (macro-exact-int? n))
         (##fail-check-exact-integer 3 extract-bit-field size position n))
        ;;;;;;;;;;;;;;;;;;;;; should check that size and position are nonnegative
        (else
         (##extract-bit-field size position n)))))

(define-prim (##test-bit-field? size position n)
  (##not (##eq? (##extract-bit-field size position n)
                0)))

(define-prim (test-bit-field? size position n)
  (macro-force-vars (size position n)
    (cond ((##not (macro-exact-int? size))
         (##fail-check-exact-integer 1 test-bit-field? size position n))
        ((##not (macro-exact-int? position))
         (##fail-check-exact-integer 2 test-bit-field? size position n))
        ((##not (macro-exact-int? n))
         (##fail-check-exact-integer 3 test-bit-field? size position n))
        ;;;;;;;;;;;;;;;;;;;;; should check that size and position are nonnegative
        (else
         (##test-bit-field? size position n)))))

(define-prim (##clear-bit-field size position n)
  (cond ((##not (macro-exact-int? size))
         (##fail-check-exact-integer 1 clear-bit-field size position n))
        ((##not (macro-exact-int? position))
         (##fail-check-exact-integer 2 clear-bit-field size position n))
        ((##not (macro-exact-int? n))
         (##fail-check-exact-integer 3 clear-bit-field size position n))
        (else
         (##replace-bit-field size position 0 n))))

(define-prim (clear-bit-field size position n)
  (macro-force-vars (size position n)
    (##clear-bit-field size position n)))

(define-prim (##replace-bit-field size position newfield n)
  (cond ((##not (macro-exact-int? size))
         (##fail-check-exact-integer 1 replace-bit-field size position newfield n))
        ((##not (macro-exact-int? position))
         (##fail-check-exact-integer 2 replace-bit-field size position newfield n))
        ((##not (macro-exact-int? newfield))
         (##fail-check-exact-integer 3 replace-bit-field size position newfield n))
        ((##not (macro-exact-int? n))
         (##fail-check-exact-integer 4 replace-bit-field size position newfield n))
        (else
         (let ((m (##bit-mask size)))
           (##bitwise-ior
            (##bitwise-and n (##bitwise-not (##arithmetic-shift m position)))
            (##arithmetic-shift (##bitwise-and newfield m) position))))))

(define-prim (replace-bit-field size position newfield n)
  (macro-force-vars (size position newfield n)
    (##replace-bit-field size position newfield n)))

(define-prim (##copy-bit-field size position from to)
  (cond ((##not (macro-exact-int? size))
         (##fail-check-exact-integer 1 copy-bit-field size position from to))
        ((##not (macro-exact-int? position))
         (##fail-check-exact-integer 2 copy-bit-field size position from to))
        ((##not (macro-exact-int? from))
         (##fail-check-exact-integer 3 copy-bit-field size position from to))
        ((##not (macro-exact-int? to))
         (##fail-check-exact-integer 4 copy-bit-field size position from to))
        (else
         (##bitwise-merge
          (##arithmetic-shift (##bit-mask size) position)
          to
          from))))

(define-prim (copy-bit-field size position from to)
  (macro-force-vars (size position from to)
    (##copy-bit-field size position from to)))

(define-prim (##bit-mask size)
  (##bitwise-not (##arithmetic-shift -1 size)))

;------------------------------------------------------------------------------

; Fixnum operations
; -----------------

(define-prim-nary (##fixnum.max x y)
  ()
  x
  (##fixnum.max x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##fixnum.min x y)
  ()
  x
  (##fixnum.min x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##fixnum.+ x y)
  0
  x
  (##fixnum.+ x y)
  macro-no-force
  macro-no-check)

(define-prim (##fixnum.+? x y))

(define-prim-nary (##fixnum.* x y)
  1
  x
  (##fixnum.* x y)
  macro-no-force
  macro-no-check)

(define-prim (##fixnum.*? x y))

(define-prim-nary (##fixnum.- x y)
  ()
  (##fixnum.- x)
  (##fixnum.- x y)
  macro-no-force
  macro-no-check)

(define-prim (##fixnum.-? x #!optional (y (macro-absent-obj)))
  (if (##eq? y (macro-absent-obj))
    (##fixnum.-? x)
    (##fixnum.-? x y)))

(define-prim (##fixnum.quotient x y))

(define-prim (##fixnum.remainder x y))

(define-prim (##fixnum.modulo x y)
  (let ((r (##fixnum.remainder x y)))
    (if (##fixnum.zero? r)
      0
      (if (##eq? (##fixnum.negative? x) (##fixnum.negative? y))
        r
        (##fixnum.+ r y)))))

(define-prim-nary (##fixnum.bitwise-ior x y)
  0
  x
  (##fixnum.bitwise-ior x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##fixnum.bitwise-xor x y)
  0
  x
  (##fixnum.bitwise-xor x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##fixnum.bitwise-and x y)
  -1
  x
  (##fixnum.bitwise-and x y)
  macro-no-force
  macro-no-check)

(define-prim (##fixnum.bitwise-not x)
  (##fixnum.- -1 x))

(define-prim (##fixnum.arithmetic-shift x y))
(define-prim (##fixnum.arithmetic-shift-left x y))
(define-prim (##fixnum.arithmetic-shift-right x y))
(define-prim (##fixnum.logical-shift-right x y))

(define-prim (##fixnum.zero? x)
  (##eq? x 0))

(define-prim (##fixnum.positive? x)
  (##fixnum.< 0 x))

(define-prim (##fixnum.negative? x)
  (##fixnum.< x 0))

(define-prim (##fixnum.odd? x)
  (##fixnum.= (##fixnum.bitwise-and x 1) 1))

(define-prim (##fixnum.even? x)
  (##fixnum.= (##fixnum.bitwise-and x 1) 0))

(define-prim-nary-bool (##fixnum.= x y)
  #t
  #t
  (##fixnum.= x y)
  macro-no-force
  macro-no-check)

(define-prim-nary-bool (##fixnum.< x y)
  #t
  #t
  (##fixnum.< x y)
  macro-no-force
  macro-no-check)

(define-prim-nary-bool (##fixnum.> x y)
  #t
  #t
  (##fixnum.< y x)
  macro-no-force
  macro-no-check)

(define-prim-nary-bool (##fixnum.<= x y)
  #t
  #t
  (##not (##fixnum.< y x))
  macro-no-force
  macro-no-check)

(define-prim-nary-bool (##fixnum.>= x y)
  #t
  #t
  (##not (##fixnum.< x y))
  macro-no-force
  macro-no-check)

(define-prim (##fixnum.->char x))
(define-prim (##fixnum.<-char x))

;------------------------------------------------------------------------------

; Bignum operations
; -----------------

; The bignum operations were mostly implemented by the "ber numerical
; analyst" Brad Lucier (http://www.math.purdue.edu/~lucier) with some
; coding guidance from Marc Feeley.

; Bignums are represented with 'adigit' vectors.  Each element is an
; integer containing ##bignum.adigit-width bits (typically 64 bits).
; These bits encode an integer in two's complement representation.
; The first element contains the least significant bits and the most
; significant bit of the last element is the sign (0=positive,
; 1=negative).

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define-prim (##bignum.negative? x))
(define-prim (##bignum.adigit-length x))
(define-prim (##bignum.adigit-inc! x i))
(define-prim (##bignum.adigit-dec! x i))
(define-prim (##bignum.adigit-add! x i y j carry))
(define-prim (##bignum.adigit-sub! x i y j borrow))
(define-prim (##bignum.mdigit-length x))
(define-prim (##bignum.mdigit-ref x i))
(define-prim (##bignum.mdigit-set! x i mdigit))
(define-prim (##bignum.mdigit-mul! x i y j multiplier carry))
(define-prim (##bignum.mdigit-div! x i y j quotient borrow))
(define-prim (##bignum.mdigit-quotient u j v_n-1))
(define-prim (##bignum.mdigit-remainder u j v_n-1 q-hat))
(define-prim (##bignum.mdigit-test? q-hat v_n-2 r-hat u_j-2))

(define-prim (##bignum.adigit-ones? x i))
(define-prim (##bignum.adigit-zero? x i))
(define-prim (##bignum.adigit-negative? x i))
(define-prim (##bignum.adigit-= x y i))
(define-prim (##bignum.adigit-< x y i))
(define-prim (##bignum.->fixnum x))
(define-prim (##bignum.<-fixnum x))
(define-prim (##bignum.adigit-shrink! x n))
(define-prim (##bignum.adigit-copy! x i y j))
(define-prim (##bignum.adigit-cat! x i hi j lo k divider))
(define-prim (##bignum.adigit-bitwise-and! x i y j))
(define-prim (##bignum.adigit-bitwise-ior! x i y j))
(define-prim (##bignum.adigit-bitwise-xor! x i y j))
(define-prim (##bignum.adigit-bitwise-not! x i))

(define-prim (##bignum.fdigit-length x))
(define-prim (##bignum.fdigit-ref x i))
(define-prim (##bignum.fdigit-set! x i fdigit))

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Bignum related constants.

(define ##bignum.adigit-ones #xffffffffffffffff)
(define ##bignum.adigit-zeros #x10000000000000000)

(define ##bignum.fdigit-base
  (##fixnum.arithmetic-shift-left 1 ##bignum.fdigit-width))

(define ##bignum.mdigit-base
  (##fixnum.arithmetic-shift-left 1 ##bignum.mdigit-width))

(define ##bignum.inexact-mdigit-base
  (##flonum.<-fixnum ##bignum.mdigit-base))

(define ##bignum.mdigit-base-minus-1
  (##fixnum.- ##bignum.mdigit-base 1))

(define ##bignum.minus-mdigit-base
  (##fixnum.- ##bignum.mdigit-base))

(define ##bignum.max-fixnum-div-mdigit-base
  (##fixnum.quotient ##max-fixnum ##bignum.mdigit-base))

(define ##bignum.min-fixnum-div-mdigit-base
  (##fixnum.quotient ##min-fixnum ##bignum.mdigit-base))

(define ##bignum.2*min-fixnum
  (if (##fixnum? -1073741824)
    -4611686018427387904 ; (- (expt 2 62))
    -1073741824))        ; (- (expt 2 30))

; The following global variables control when each of the three
; multiplication algorithms are used.

(define ##bignum.naive-mul-max-width 1400)
(set! ##bignum.naive-mul-max-width ##bignum.naive-mul-max-width)

(define ##bignum.fft-mul-min-width 6800)
(set! ##bignum.fft-mul-min-width ##bignum.fft-mul-min-width)

#|
;; see the wonderful paper
;; "Rapid multiplication modulo the sum and difference of highly
;; composite numbers" by Colin Percival, electronically published
;; by Mathematics of Computation, number S 0025-5718(02)01419-9, URL
;; http://www.ams.org/journal-getitem?pii=S0025-5718-02-01419-9
;; that gives these very nice error bounds.  This should be published
;; in the paper journal sometime after March 2002.

(define epsilon (expt 2. -53))
(define bigepsilon (* epsilon (sqrt 5)))
(define n 27)
(define beta (expt 2. -53))      ;; accuracy of trigonometric inputs
(define l 8)
(define norm-x (sqrt (* (expt 2 n) (* (expt 2 l)
                                      (expt 2 l))))) ;; overestimate by 0.5%
(define norm-y norm-x)
(define error (* norm-x
                 norm-y
                 ;; the following three lines use the slight overestimate that
                 ;; ln(1+epsilon) = epsilon, etc.
                 ;; there are more accurate ways to calculate this, but we
                 ;; don't really need them.
                 (- (* (exp (* 3 n epsilon))
                       (exp (* (+ (* 3 n) 1) bigepsilon))
                       (exp (* beta 3 n)))
                    1)))
(pp error)
;; error is .33593750000000006 < 1/2
;; so if we wanted to, we could have a result of 2^27 8-bit digits output,
;; i.e., 2^26 8-bit digits input or 536870912 bits input.

;; But we don't have great control over the accuracy of the trigonometric
;; inputs, so let's see what happens if we quadruple the bound on beta and
;; reduce n to 26

(define n 26)
(define norm-x (sqrt (* (expt 2 n) (* (expt 2 l)
                                      (expt 2 l)))))
(define norm-y norm-x)

(define beta (expt 2. -51))
(define error (* norm-x
                 norm-y
                 (- (* (exp (* 3 n epsilon))
                       (exp (* (+ (* 3 n) 1) bigepsilon))
                       (exp (* beta 3 n)))
                    1)))

(pp error)

;; error bound is .2763671875 < 1/2
;; so it's no problem to have 2^26 8-bit digits output, or 2^25 eight-bit digits
;; input, or 268435456 bits input.  This requires 2*2^26 64-bit doubles,
;; or 2^30 bytes or 1 Gbyte to do the multiplication.

;; let's look at what happens when you have 4-bit fft words:

(define epsilon (expt 2. -53))
(define bigepsilon (* epsilon (sqrt 5)))
(define n 35)
(define beta (expt 2. -53))      ;; accuracy of trigonometric inputs
(define l 4)
(define norm-x (sqrt (* (expt 2 n) (* (expt 2 l)
                                      (expt 2 l))))) ;; overestimate by 0.5%
(define norm-y norm-x)
(define error (* norm-x
                 norm-y
                 ;; the following three lines use the slight overestimate that
                 ;; ln(1+epsilon) = epsilon, etc.
                 ;; there are more accurate ways to calculate this, but we
                 ;; don't really need them.
                 (- (* (exp (* 3 n epsilon))
                       (exp (* (+ (* 3 n) 1) bigepsilon))
                       (exp (* beta 3 n)))
                    1)))
(pp error)

;; this is good, you can have (* (expt 2 34) 4) = 68,719,476,736 bits input.
;; now you need 32 times the memory of the input numbers to run, but hey, everyone
;; has 128GB of ram when you need it, right?

;; I think I won't add a four-bit fft word option for now.

;; Because the fft algorithm as written requires temporary storage up to
;; sixteen times the size of the final result, people working with large
;; integers but barely enough memory on 64-bit machines may wish to
;; set! ##bignum.fft-mul-max-width to a slightly smaller value so that
;; karatsuba will kick in earlier for the largest integers.

|#

(define ##bignum.fft-mul-max-width
  (if (##fixnum? -1073741824) ; to avoid creating f64vectors that are too long
    268435456
    4194304))
(set! ##bignum.fft-mul-max-width ##bignum.fft-mul-max-width)


(define ##bignum.fast-gcd-size ##bignum.fft-mul-min-width)  ;; must be >= 64
(set! ##bignum.fast-gcd-size ##bignum.fast-gcd-size)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; Operations where arguments are in bignum format

(define-prim (##bignum.make k x complement?)
  (##declare (not interrupts-enabled))
  (let ((v (##c-code "
long i;
long n = ___INT(___ARG1);
#if ___BIG_ABASE_WIDTH == 32
long words = ___WORDS((n*(___BIG_ABASE_WIDTH/8))) + 1;
#else
#if ___WS == 4
long words = ___WORDS((n*(___BIG_ABASE_WIDTH/8))) + 2;
#else
long words = ___WORDS((n*(___BIG_ABASE_WIDTH/8))) + 1;
#endif
#endif
___SCMOBJ result;

if (n > (long)(unsigned long)(___LMASK>>___LF)/(___BIG_ABASE_WIDTH/8))
  result = ___FIX(___HEAP_OVERFLOW_ERR); /* requested object is too big! */
else if (words > ___MSECTION_BIGGEST)
  {
    ___FRAME_STORE_RA(___R0)
    ___W_ALL
#if ___BIG_ABASE_WIDTH == 32
    result = ___alloc_scmobj (___sBIGNUM, n<<2, ___STILL);
#else
    result = ___alloc_scmobj (___sBIGNUM, n<<3, ___STILL);
#endif
    ___R_ALL
    ___SET_R0(___FRAME_FETCH_RA)
    if (!___FIXNUMP(result))
      ___still_obj_refcount_dec (result);
  }
else
  {
    ___BOOL overflow = 0;
    ___hp += words;
    if (___hp > ___ps->heap_limit)
      {
        ___FRAME_STORE_RA(___R0)
        ___W_ALL
        overflow = ___heap_limit () && ___garbage_collect (0);
        ___R_ALL
        ___SET_R0(___FRAME_FETCH_RA)
      }
    else
      ___hp -= words;
    if (overflow)
      result = ___FIX(___HEAP_OVERFLOW_ERR);
    else
      {
#if ___BIG_ABASE_WIDTH == 32
        result = ___TAG(___hp, ___tSUBTYPED);
#else
#if ___WS == 4
        result = ___TAG(___CAST(___SCMOBJ*,___CAST(___SCMOBJ,___hp+2)&~7)-1,
                        ___tSUBTYPED);
#else
        result = ___TAG(___hp, ___tSUBTYPED);
#endif
#endif
#if ___BIG_ABASE_WIDTH == 32
        ___HEADER(result) = ___MAKE_HD_BYTES((n<<2), ___sBIGNUM);
#else
        ___HEADER(result) = ___MAKE_HD_BYTES((n<<3), ___sBIGNUM);
#endif
        ___hp += words;
      }
  }
if (!___FIXNUMP(result))
  {
    ___SCMOBJ x = ___ARG2;
    ___SCMOBJ len;
    if (x == ___FAL)
      len = 0;
    else
      {
        len = ___INT(___BIGALENGTH(x));
        if (len > n)
          len = n;
      }
#if ___BIG_ABASE_WIDTH == 32
    if (___ARG3 == ___FAL)
      {
        for (i=0; i<len; i++)
          ___STORE_U32(___BODY_AS(result,___tSUBTYPED),i,
                       ___FETCH_U32(___BODY_AS(x,___tSUBTYPED),i));
        if (x != ___FAL &&
            ___FETCH_S32(___BODY_AS(x,___tSUBTYPED),(i-1)) < 0)
          for (; i<n; i++)
            ___STORE_U32(___BODY_AS(result,___tSUBTYPED),i,___BIG_ABASE_MIN_1);
        else
          for (; i<n; i++)
            ___STORE_U32(___BODY_AS(result,___tSUBTYPED),i,0);
      }
    else
      {
        for (i=0; i<len; i++)
          ___STORE_U32(___BODY_AS(result,___tSUBTYPED),i,
                       ~___FETCH_U32(___BODY_AS(x,___tSUBTYPED),i));
        if (x != ___FAL &&
            ___FETCH_S32(___BODY_AS(x,___tSUBTYPED),(i-1)) < 0)
          for (; i<n; i++)
            ___STORE_U32(___BODY_AS(result,___tSUBTYPED),i,0);
        else
          for (; i<n; i++)
            ___STORE_U32(___BODY_AS(result,___tSUBTYPED),i,___BIG_ABASE_MIN_1);
      }
#else
    if (___ARG3 == ___FAL)
      {
        for (i=0; i<len; i++)
          ___STORE_U64(___BODY_AS(result,___tSUBTYPED),i,
                       ___FETCH_U64(___BODY_AS(x,___tSUBTYPED),i));
        if (x != ___FAL &&
            ___FETCH_S64(___BODY_AS(x,___tSUBTYPED),(i-1)) < 0)
          for (; i<n; i++)
            ___STORE_U64(___BODY_AS(result,___tSUBTYPED),i,___BIG_ABASE_MIN_1);
        else
          for (; i<n; i++)
            ___STORE_U64(___BODY_AS(result,___tSUBTYPED),i,0);
      }
    else
      {
        for (i=0; i<len; i++)
          ___STORE_U64(___BODY_AS(result,___tSUBTYPED),i,
                       ~___FETCH_U64(___BODY_AS(x,___tSUBTYPED),i));
        if (x != ___FAL &&
            ___FETCH_S64(___BODY_AS(x,___tSUBTYPED),(i-1)) < 0)
          for (; i<n; i++)
            ___STORE_U64(___BODY_AS(result,___tSUBTYPED),i,0);
        else
          for (; i<n; i++)
            ___STORE_U64(___BODY_AS(result,___tSUBTYPED),i,___BIG_ABASE_MIN_1);
      }
#endif
  }
___RESULT = result;
" k x complement?)))
    (if (##fixnum? v)
      (begin
        (##raise-heap-overflow-exception)
        (##bignum.make k x complement?))
      v)))

; Bignum comparison.

(define-prim (##bignum.= x y)

  ; x is a normalized bignum, y is a normalized bignum

  (##declare (not interrupts-enabled))

  (let ((x-length (##bignum.adigit-length x)))
    (and (##fixnum.= x-length (##bignum.adigit-length y))
         (let loop ((i (##fixnum.- x-length 1)))
           (or (##fixnum.< i 0)
               (and (##bignum.adigit-= x y i)
                    (loop (##fixnum.- i 1))))))))

(define-prim (##bignum.< x y)

  ; x is a normalized bignum, y is a normalized bignum

  (##declare (not interrupts-enabled))

  (define (loop i)
    (and (##not (##fixnum.< i 0))
         (or (##bignum.adigit-< x y i)
             (and (##bignum.adigit-= x y i)
                  (loop (##fixnum.- i 1))))))

  (if (##bignum.negative? x)
    (if (##bignum.negative? y)
      (let ((x-length (##bignum.adigit-length x))
            (y-length (##bignum.adigit-length y)))
        (or (##fixnum.< y-length x-length)
            (and (##fixnum.= x-length y-length)
                 (loop (##fixnum.- x-length 1)))))
      #t)
    (if (##bignum.negative? y)
      #f
      (let ((x-length (##bignum.adigit-length x))
            (y-length (##bignum.adigit-length y)))
        (or (##fixnum.< x-length y-length)
            (and (##fixnum.= x-length y-length)
                 (loop (##fixnum.- x-length 1))))))))

; Bignum addition and subtraction.

(define-prim (##bignum.+ x y)

  ; x is an unnormalized bignum, y is an unnormalized bignum

  (define (add x x-length y y-length)
    (let* ((result-length
            (##fixnum.+ y-length
                        (if (##eq? (##bignum.negative? x)
                                   (##bignum.negative? y))
                          1
                          0)))
           (result
            (##bignum.make result-length y #f)))

      (##declare (not interrupts-enabled))

      (let loop ((i 0)
                 (carry 0))
        (if (##fixnum.< i x-length)
          (loop (##fixnum.+ i 1)
                (##bignum.adigit-add! result i x i carry))
          (##bignum.propagate-carry-and-normalize!
           result
           result-length
           x-length
           (##bignum.negative? x)
           (##fixnum.zero? carry))))))

  (let ((x-length (##bignum.adigit-length x))
        (y-length (##bignum.adigit-length y)))
    (if (##fixnum.< x-length y-length)
      (add x x-length y y-length)
      (add y y-length x x-length))))

(define-prim (##bignum.- x y)

  ; x is an unnormalized bignum, y is an unnormalized bignum

  (let ((x-length (##bignum.adigit-length x))
        (y-length (##bignum.adigit-length y)))
    (if (##fixnum.< x-length y-length)

      (let* ((result-length
              (##fixnum.+ y-length
                          (if (##eq? (##bignum.negative? x)
                                     (##bignum.negative? y))
                            0
                            1)))
             (result
              (##bignum.make result-length y #t)))

        (##declare (not interrupts-enabled))

        (let loop1 ((i 0)
                    (carry 1))
          (if (##fixnum.< i x-length)
            (loop1 (##fixnum.+ i 1)
                   (##bignum.adigit-add! result i x i carry))
            (##bignum.propagate-carry-and-normalize!
             result
             result-length
             x-length
             (##bignum.negative? x)
             (##fixnum.zero? carry)))))

      (let* ((result-length
              (##fixnum.+ x-length
                          (if (##eq? (##bignum.negative? x)
                                     (##bignum.negative? y))
                            0
                            1)))
             (result
              (##bignum.make result-length x #f)))

        (##declare (not interrupts-enabled))

        (let loop2 ((i 0)
                    (borrow 0))
          (if (##fixnum.< i y-length)
            (loop2 (##fixnum.+ i 1)
                   (##bignum.adigit-sub! result i y i borrow))
            (##bignum.propagate-carry-and-normalize!
             result
             result-length
             y-length
             (##not (##bignum.negative? y))
             (##not (##fixnum.zero? borrow)))))))))

(define-prim (##bignum.propagate-carry-and-normalize!
              result
              result-length
              i
              borrow?
              propagate?)

  (##declare (not interrupts-enabled))

  (if (##eq? borrow? propagate?)
    (if borrow?

      (let loop1 ((i i)
                  (borrow 1))
        (if (and (##not (##fixnum.zero? borrow))
                 (##fixnum.< i result-length))
          (loop1 (##fixnum.+ i 1)
                 (##bignum.adigit-dec! result i))
          (##bignum.normalize! result)))

      (let loop2 ((i i)
                  (carry 1))
        (if (and (##not (##fixnum.zero? carry))
                 (##fixnum.< i result-length))
          (loop2 (##fixnum.+ i 1)
                 (##bignum.adigit-inc! result i))
          (##bignum.normalize! result))))

    (##bignum.normalize! result)))

(define-prim (##bignum.normalize! result)

  (##declare (not interrupts-enabled))

  (let ((n (##fixnum.- (##bignum.adigit-length result) 1)))

    (cond ((##bignum.adigit-zero? result n)
           (let loop1 ((i (##fixnum.- n 1)))
             (cond ((##fixnum.< i 0)
                    0)
                   ((##bignum.adigit-zero? result i)
                    (loop1 (##fixnum.- i 1)))
                   ((##bignum.adigit-negative? result i)
                    (##bignum.adigit-shrink! result (##fixnum.+ i 2)))
                   (else
                    (or (and (##fixnum.= i 0)
                             (##bignum.->fixnum result))
                        (##bignum.adigit-shrink! result (##fixnum.+ i 1)))))))

          ((##bignum.adigit-ones? result n)
           (let loop2 ((i (##fixnum.- n 1)))
             (cond ((##fixnum.< i 0)
                    -1)
                   ((##bignum.adigit-ones? result i)
                    (loop2 (##fixnum.- i 1)))
                   ((##not (##bignum.adigit-negative? result i))
                    (##bignum.adigit-shrink! result (##fixnum.+ i 2)))
                   (else
                    (or (and (##fixnum.= i 0)
                             (##bignum.->fixnum result))
                        (##bignum.adigit-shrink! result (##fixnum.+ i 1)))))))

          ((and (##fixnum.= n 0)
                (##bignum.->fixnum result))
           =>
           (lambda (x) x))

          (else
           result))))

; Bignum multiplication.

(define-prim (##bignum.* x y)

  ; Routines for real FFT translated from Ooura's code.

  (define (makewt nw ip ipcontrol w)
    (##vector-set! ipcontrol 0 nw)
    (##vector-set! ipcontrol 1 1)
    (let* ((nwh
            (##fixnum.arithmetic-shift-right nw 1))
           (delta
            (##flonum./ (##flonum.atan (macro-inexact-+1))
                        (##flonum.<-fixnum nwh))))
      (##f64vector-set! w 0 (macro-inexact-+1))
      (##f64vector-set! w 1 (macro-inexact-+0))
      (##f64vector-set! w nwh
        (##flonum.cos (##flonum.* delta (##flonum.<-fixnum nwh))))
      (##f64vector-set! w (##fixnum.+ nwh 1)
        (##f64vector-ref w nwh))
      (let loop ((j 2))
        (##declare (not interrupts-enabled))
        (if (##fixnum.< j nwh)
          (let ((arg (##flonum.* delta (##flonum.<-fixnum j))))
            (let ((x (##flonum.cos arg))
                  (y (##flonum.sin arg)))
              (##f64vector-set! w j x)
              (##f64vector-set! w (##fixnum.+ j 1) y)
              (##f64vector-set! w (##fixnum.- nw j) y)
              (##f64vector-set! w (##fixnum.- (##fixnum.+ nw 1) j) x)
              (loop (##fixnum.+ j 2))))
          (bitrv2 nw ip w)))))

  (define (makect nc ipcontrol c)
    (##vector-set! ipcontrol 1 nc)
    (let* ((nch
            (##fixnum.arithmetic-shift-right nc 1))
           (delta
            (##flonum./ (##flonum.atan (macro-inexact-+1))
                        (##flonum.<-fixnum nch))))
      (##f64vector-set! c 0
        (##flonum.cos (##flonum.* delta (##flonum.<-fixnum nch))))
      (##f64vector-set! c nch
        (##flonum.* (macro-inexact-+1/2) (##f64vector-ref c 0)))
      (let loop ((j 1))
        (##declare (not interrupts-enabled))
        (if (##fixnum.< j nch)
          (let ((arg (##flonum.* delta (##flonum.<-fixnum j))))
            (##f64vector-set! c j
              (##flonum.* (macro-inexact-+1/2) (##flonum.cos arg)))
            (##f64vector-set! c (##fixnum.- nc j)
              (##flonum.* (macro-inexact-+1/2) (##flonum.sin arg)))
            (loop (##fixnum.+ j 1)))))))

  (define (bitrv2 n ip a)

    (##declare (not interrupts-enabled))

    (##vector-set! ip 0 0)

    (let loop1 ((l n) (m 1))

      (cond ((##fixnum.< (##fixnum.arithmetic-shift-left m 2) l)
             (let ((l (##fixnum.arithmetic-shift-right l 1)))
               (let loop2 ((j 0))
                 (if (##fixnum.< j m)
                   (begin
                     (##vector-set! ip (##fixnum.+ m j)
                       (##fixnum.+ (##vector-ref ip j) l))
                     (loop2 (##fixnum.+ j 1)))
                   (loop1 l (##fixnum.arithmetic-shift-left m 1))))))

            ((##fixnum.< l (##fixnum.arithmetic-shift-left m 2))
             (let loop3 ((k 1))
               (if (##fixnum.< k m)
                 (let loop4 ((j 0))
                   (if (##fixnum.< j k)
                     (let ((j1
                            (##fixnum.+ (##fixnum.arithmetic-shift-left j 1)
                                        (##vector-ref ip k)))
                           (k1
                            (##fixnum.+ (##fixnum.arithmetic-shift-left k 1)
                                        (##vector-ref ip j))))
                       (let ((xr (##f64vector-ref a (##fixnum.+ j1 0)))
                             (xi (##f64vector-ref a (##fixnum.+ j1 1))))
                         (##f64vector-set! a (##fixnum.+ j1 0)
                           (##f64vector-ref a (##fixnum.+ k1 0)))
                         (##f64vector-set! a (##fixnum.+ j1 1)
                           (##f64vector-ref a (##fixnum.+ k1 1)))
                         (##f64vector-set! a (##fixnum.+ k1 0) xr)
                         (##f64vector-set! a (##fixnum.+ k1 1) xi))
                       (loop4 (##fixnum.+ j 1)))
                     (loop3 (##fixnum.+ k 1)))))))

            (else
             (let ((m2 (##fixnum.arithmetic-shift-left m 1)))
               (let loop5 ((k 1))
                 (if (##fixnum.< k m)
                   (let loop6 ((j 0))
                     (if (##fixnum.< j k)
                       (let ((j1
                              (##fixnum.+ (##fixnum.arithmetic-shift-left j 1)
                                          (##vector-ref ip k)))
                             (k1
                              (##fixnum.+ (##fixnum.arithmetic-shift-left k 1)
                                          (##vector-ref ip j))))
                         (let ((xr (##f64vector-ref a (##fixnum.+ j1 0)))
                               (xi (##f64vector-ref a (##fixnum.+ j1 1))))
                           (##f64vector-set! a (##fixnum.+ j1 0)
                             (##f64vector-ref a (##fixnum.+ k1 0)))
                           (##f64vector-set! a (##fixnum.+ j1 1)
                             (##f64vector-ref a (##fixnum.+ k1 1)))
                           (##f64vector-set! a (##fixnum.+ k1 0) xr)
                           (##f64vector-set! a (##fixnum.+ k1 1) xi))
                         (let ((j1 (##fixnum.+ j1 m2))
                               (k1 (##fixnum.+ k1 m2)))
                           (let ((xr (##f64vector-ref a (##fixnum.+ j1 0)))
                                 (xi (##f64vector-ref a (##fixnum.+ j1 1))))
                             (##f64vector-set! a (##fixnum.+ j1 0)
                               (##f64vector-ref a (##fixnum.+ k1 0)))
                             (##f64vector-set! a (##fixnum.+ j1 1)
                               (##f64vector-ref a (##fixnum.+ k1 1)))
                             (##f64vector-set! a (##fixnum.+ k1 0) xr)
                             (##f64vector-set! a (##fixnum.+ k1 1) xi)))
                         (loop6 (##fixnum.+ j 1)))
                       (loop5 (##fixnum.+ k 1)))))))))))

  (define (cftfsub n a w)

    (define (cleanup l)

      (##declare (not interrupts-enabled))

      (if (##fixnum.= (##fixnum.arithmetic-shift-left l 2) n)

        (let loop1 ((j 0))
          (if (##fixnum.< j l)
            (let* ((j1 (##fixnum.+ j  l))
                   (j2 (##fixnum.+ j1 l))
                   (j3 (##fixnum.+ j2 l)))
              (let ((aj00 (##f64vector-ref a (##fixnum.+ j  0)))
                    (aj01 (##f64vector-ref a (##fixnum.+ j  1)))
                    (aj10 (##f64vector-ref a (##fixnum.+ j1 0)))
                    (aj11 (##f64vector-ref a (##fixnum.+ j1 1)))
                    (aj20 (##f64vector-ref a (##fixnum.+ j2 0)))
                    (aj21 (##f64vector-ref a (##fixnum.+ j2 1)))
                    (aj30 (##f64vector-ref a (##fixnum.+ j3 0)))
                    (aj31 (##f64vector-ref a (##fixnum.+ j3 1))))
                (let ((x0r (##flonum.+ aj00 aj10))
                      (x0i (##flonum.+ aj01 aj11))
                      (x1r (##flonum.- aj00 aj10))
                      (x1i (##flonum.- aj01 aj11))
                      (x2r (##flonum.+ aj20 aj30))
                      (x2i (##flonum.+ aj21 aj31))
                      (x3r (##flonum.- aj20 aj30))
                      (x3i (##flonum.- aj21 aj31)))
                  (##f64vector-set! a (##fixnum.+ j  0) (##flonum.+ x0r x2r))
                  (##f64vector-set! a (##fixnum.+ j  1) (##flonum.+ x0i x2i))
                  (##f64vector-set! a (##fixnum.+ j2 0) (##flonum.- x0r x2r))
                  (##f64vector-set! a (##fixnum.+ j2 1) (##flonum.- x0i x2i))
                  (##f64vector-set! a (##fixnum.+ j1 0) (##flonum.- x1r x3i))
                  (##f64vector-set! a (##fixnum.+ j1 1) (##flonum.+ x1i x3r))
                  (##f64vector-set! a (##fixnum.+ j3 0) (##flonum.+ x1r x3i))
                  (##f64vector-set! a (##fixnum.+ j3 1) (##flonum.- x1i x3r))))
              (loop1 (##fixnum.+ j 2)))))

        (let loop2 ((j 0))
          (if (##fixnum.< j l)
            (let ((j1 (##fixnum.+ j l)))
              (let ((aj00 (##f64vector-ref a (##fixnum.+ j  0)))
                    (aj01 (##f64vector-ref a (##fixnum.+ j  1)))
                    (aj10 (##f64vector-ref a (##fixnum.+ j1 0)))
                    (aj11 (##f64vector-ref a (##fixnum.+ j1 1))))
                (let ((x0r (##flonum.- aj00 aj10))
                      (x0i (##flonum.- aj01 aj11)))
                  (##f64vector-set! a (##fixnum.+ j 0) (##flonum.+ aj00 aj10))
                  (##f64vector-set! a (##fixnum.+ j 1) (##flonum.+ aj01 aj11))
                  (##f64vector-set! a (##fixnum.+ j1 0) x0r)
                  (##f64vector-set! a (##fixnum.+ j1 1) x0i)))
              (loop2 (##fixnum.+ j 2)))))))

    (cft1st n a w)
    (let loop ((l 8))
      (if (##fixnum.< (##fixnum.arithmetic-shift-left l 2) n)
        (begin
          (cftmdl n l a w)
          (loop (##fixnum.arithmetic-shift-left l 2)))
        (cleanup l))))

  (define (cftbsub n a w)

    (define (cleanup l)

      (##declare (not interrupts-enabled))

      (if (##fixnum.= (##fixnum.arithmetic-shift-left l 2) n)

        (let loop1 ((j 0))
          (if (##fixnum.< j l)
            (let* ((j1 (##fixnum.+ j  l))
                   (j2 (##fixnum.+ j1 l))
                   (j3 (##fixnum.+ j2 l)))
              (let ((aj00 (##f64vector-ref a (##fixnum.+ j  0)))
                    (aj01 (##f64vector-ref a (##fixnum.+ j  1)))
                    (aj10 (##f64vector-ref a (##fixnum.+ j1 0)))
                    (aj11 (##f64vector-ref a (##fixnum.+ j1 1)))
                    (aj20 (##f64vector-ref a (##fixnum.+ j2 0)))
                    (aj21 (##f64vector-ref a (##fixnum.+ j2 1)))
                    (aj30 (##f64vector-ref a (##fixnum.+ j3 0)))
                    (aj31 (##f64vector-ref a (##fixnum.+ j3 1))))
                (let ((x0r             (##flonum.+ aj00 aj10))
                      (x0i (##flonum.- (##flonum.+ aj01 aj11)))
                      (x1r             (##flonum.- aj00 aj10))
                      (x1i (##flonum.- (##flonum.- aj01 aj11)))
                      (x2r             (##flonum.+ aj20 aj30))
                      (x2i             (##flonum.+ aj21 aj31))
                      (x3r             (##flonum.- aj20 aj30))
                      (x3i             (##flonum.- aj21 aj31)))
                  (##f64vector-set! a (##fixnum.+ j  0) (##flonum.+ x0r x2r))
                  (##f64vector-set! a (##fixnum.+ j  1) (##flonum.- x0i x2i))
                  (##f64vector-set! a (##fixnum.+ j2 0) (##flonum.- x0r x2r))
                  (##f64vector-set! a (##fixnum.+ j2 1) (##flonum.+ x0i x2i))
                  (##f64vector-set! a (##fixnum.+ j1 0) (##flonum.- x1r x3i))
                  (##f64vector-set! a (##fixnum.+ j1 1) (##flonum.- x1i x3r))
                  (##f64vector-set! a (##fixnum.+ j3 0) (##flonum.+ x1r x3i))
                  (##f64vector-set! a (##fixnum.+ j3 1) (##flonum.+ x1i x3r))))
              (loop1 (##fixnum.+ j 2)))))

        (let loop2 ((j 0))
          (if (##fixnum.< j l)
            (let ((j1 (##fixnum.+ j l)))
              (let ((aj00 (##f64vector-ref a (##fixnum.+ j  0)))
                    (aj01 (##f64vector-ref a (##fixnum.+ j  1)))
                    (aj10 (##f64vector-ref a (##fixnum.+ j1 0)))
                    (aj11 (##f64vector-ref a (##fixnum.+ j1 1))))
                (let ((x0r             (##flonum.- aj00 aj10))
                      (x0i (##flonum.- (##flonum.- aj01 aj11))))
                  (##f64vector-set! a (##fixnum.+ j 0)
                    (##flonum.+ aj00 aj10))
                  (##f64vector-set! a (##fixnum.+ j 1)
                    (##flonum.- (##flonum.+ aj01 aj11)))
                  (##f64vector-set! a (##fixnum.+ j1 0) x0r)
                  (##f64vector-set! a (##fixnum.+ j1 1) x0i)))
              (loop2 (##fixnum.+ j 2)))))))

    (cft1st n a w)
    (let loop ((l 8))
      (if (##fixnum.< (##fixnum.arithmetic-shift-left l 2) n)
        (begin
          (cftmdl n l a w)
          (loop (##fixnum.arithmetic-shift-left l 2)))
        (cleanup l))))

  (define (cft1st n a w)

    (##declare (not interrupts-enabled))

    (let ((a0 (##f64vector-ref a 0))
          (a1 (##f64vector-ref a 1))
          (a2 (##f64vector-ref a 2))
          (a3 (##f64vector-ref a 3))
          (a4 (##f64vector-ref a 4))
          (a5 (##f64vector-ref a 5))
          (a6 (##f64vector-ref a 6))
          (a7 (##f64vector-ref a 7)))
      (let ((x0r (##flonum.+ a0 a2))
            (x0i (##flonum.+ a1 a3))
            (x1r (##flonum.- a0 a2))
            (x1i (##flonum.- a1 a3))
            (x2r (##flonum.+ a4 a6))
            (x2i (##flonum.+ a5 a7))
            (x3r (##flonum.- a4 a6))
            (x3i (##flonum.- a5 a7)))
        (##f64vector-set! a 0 (##flonum.+ x0r x2r))
        (##f64vector-set! a 1 (##flonum.+ x0i x2i))
        (##f64vector-set! a 4 (##flonum.- x0r x2r))
        (##f64vector-set! a 5 (##flonum.- x0i x2i))
        (##f64vector-set! a 2 (##flonum.- x1r x3i))
        (##f64vector-set! a 3 (##flonum.+ x1i x3r))
        (##f64vector-set! a 6 (##flonum.+ x1r x3i))
        (##f64vector-set! a 7 (##flonum.- x1i x3r))))
    (let ((a8 (##f64vector-ref a 8))
          (a9 (##f64vector-ref a 9))
          (a10 (##f64vector-ref a 10))
          (a11 (##f64vector-ref a 11))
          (a12 (##f64vector-ref a 12))
          (a13 (##f64vector-ref a 13))
          (a14 (##f64vector-ref a 14))
          (a15 (##f64vector-ref a 15)))
      (let ((wk1r (##f64vector-ref w 2))
            (x0r (##flonum.+ a8 a10))
            (x0i (##flonum.+ a9 a11))
            (x1r (##flonum.- a8 a10))
            (x1i (##flonum.- a9 a11))
            (x2r (##flonum.+ a12 a14))
            (x2i (##flonum.+ a13 a15))
            (x3r (##flonum.- a12 a14))
            (x3i (##flonum.- a13 a15)))
        (##f64vector-set! a 8 (##flonum.+ x0r x2r))
        (##f64vector-set! a 9 (##flonum.+ x0i x2i))
        (##f64vector-set! a 12 (##flonum.- x2i x0i))
        (##f64vector-set! a 13 (##flonum.- x0r x2r))
        (let ((x0r (##flonum.- x1r x3i))
              (x0i (##flonum.+ x1i x3r)))
          (##f64vector-set! a 10 (##flonum.* wk1r (##flonum.- x0r x0i)))
          (##f64vector-set! a 11 (##flonum.* wk1r (##flonum.+ x0r x0i))))
        (let ((x0r (##flonum.+ x3i x1r))
              (x0i (##flonum.- x3r x1i)))
          (##f64vector-set! a 14 (##flonum.* wk1r (##flonum.- x0i x0r)))
          (##f64vector-set! a 15 (##flonum.* wk1r (##flonum.+ x0i x0r))))))
    (let loop ((j 16) (k1 2))
      (if (##fixnum.< j n)
        (let ((k2 (##fixnum.arithmetic-shift-left k1 1))
              (wk2r (##f64vector-ref w k1))
              (wk2i (##f64vector-ref w (##fixnum.+ k1 1))))
          (let ((wk1r (##f64vector-ref w k2))
                (wk1i (##f64vector-ref w (##fixnum.+ k2 1))))
            (let ((a0 (##f64vector-ref a (##fixnum.+ j 0)))
                  (a1 (##f64vector-ref a (##fixnum.+ j 1)))
                  (a2 (##f64vector-ref a (##fixnum.+ j 2)))
                  (a3 (##f64vector-ref a (##fixnum.+ j 3)))
                  (a4 (##f64vector-ref a (##fixnum.+ j 4)))
                  (a5 (##f64vector-ref a (##fixnum.+ j 5)))
                  (a6 (##f64vector-ref a (##fixnum.+ j 6)))
                  (a7 (##f64vector-ref a (##fixnum.+ j 7))))
              (let ((wk3r
                     (##flonum.- wk1r
                                 (##flonum.* (macro-inexact-+2)
                                             (##flonum.* wk2i wk1i))))
                    (wk3i
                     (##flonum.- (##flonum.* (macro-inexact-+2)
                                             (##flonum.* wk2i wk1r))
                                 wk1i))
                    (x0r (##flonum.+ a0 a2))
                    (x0i (##flonum.+ a1 a3))
                    (x1r (##flonum.- a0 a2))
                    (x1i (##flonum.- a1 a3))
                    (x2r (##flonum.+ a4 a6))
                    (x2i (##flonum.+ a5 a7))
                    (x3r (##flonum.- a4 a6))
                    (x3i (##flonum.- a5 a7)))
                (##f64vector-set! a (##fixnum.+ j 0) (##flonum.+ x0r x2r))
                (##f64vector-set! a (##fixnum.+ j 1) (##flonum.+ x0i x2i))
                (let ((x0r (##flonum.- x0r x2r))
                      (x0i (##flonum.- x0i x2i)))
                  (##f64vector-set! a (##fixnum.+ j 4)
                    (##flonum.- (##flonum.* wk2r x0r)
                                (##flonum.* wk2i x0i)))
                  (##f64vector-set! a (##fixnum.+ j 5)
                    (##flonum.+ (##flonum.* wk2r x0i)
                                (##flonum.* wk2i x0r))))
                (let ((x0r (##flonum.- x1r x3i))
                      (x0i (##flonum.+ x1i x3r)))
                  (##f64vector-set! a (##fixnum.+ j 2)
                    (##flonum.- (##flonum.* wk1r x0r)
                                (##flonum.* wk1i x0i)))
                  (##f64vector-set! a (##fixnum.+ j 3)
                    (##flonum.+ (##flonum.* wk1r x0i)
                                (##flonum.* wk1i x0r))))
                (let ((x0r (##flonum.+ x1r x3i))
                      (x0i (##flonum.- x1i x3r)))
                  (##f64vector-set! a (##fixnum.+ j 6)
                    (##flonum.- (##flonum.* wk3r x0r)
                                (##flonum.* wk3i x0i)))
                  (##f64vector-set! a (##fixnum.+ j 7)
                    (##flonum.+ (##flonum.* wk3r x0i)
                                (##flonum.* wk3i x0r)))))))
          (let ((wk1r (##f64vector-ref w (##fixnum.+ k2 2)))
                (wk1i (##f64vector-ref w (##fixnum.+ k2 3))))
            (let (( a8 (##f64vector-ref a (##fixnum.+ j  8)))
                  ( a9 (##f64vector-ref a (##fixnum.+ j  9)))
                  (a10 (##f64vector-ref a (##fixnum.+ j 10)))
                  (a11 (##f64vector-ref a (##fixnum.+ j 11)))
                  (a12 (##f64vector-ref a (##fixnum.+ j 12)))
                  (a13 (##f64vector-ref a (##fixnum.+ j 13)))
                  (a14 (##f64vector-ref a (##fixnum.+ j 14)))
                  (a15 (##f64vector-ref a (##fixnum.+ j 15))))
              (let ((wk3r
                     (##flonum.- wk1r
                                 (##flonum.* (macro-inexact-+2)
                                             (##flonum.* wk2r wk1i))))
                    (wk3i
                     (##flonum.- (##flonum.* (macro-inexact-+2)
                                             (##flonum.* wk2r wk1r))
                                 wk1i))
                    (x0r (##flonum.+ a8 a10))
                    (x0i (##flonum.+ a9 a11))
                    (x1r (##flonum.- a8 a10))
                    (x1i (##flonum.- a9 a11))
                    (x2r (##flonum.+ a12 a14))
                    (x2i (##flonum.+ a13 a15))
                    (x3r (##flonum.- a12 a14))
                    (x3i (##flonum.- a13 a15)))
                (##f64vector-set! a (##fixnum.+ j 8) (##flonum.+ x0r x2r))
                (##f64vector-set! a (##fixnum.+ j 9) (##flonum.+ x0i x2i))
                (let ((x0r (##flonum.- x0r x2r))
                      (x0i (##flonum.- x0i x2i)))
                  (##f64vector-set! a (##fixnum.+ j 12)
                    (##flonum.- (##flonum.+ (##flonum.* wk2i x0r)
                                            (##flonum.* wk2r x0i))))
                  (##f64vector-set! a (##fixnum.+ j 13)
                    (##flonum.- (##flonum.- (##flonum.* wk2i x0i)
                                            (##flonum.* wk2r x0r)))))
                (let ((x0r (##flonum.- x1r x3i))
                      (x0i (##flonum.+ x1i x3r)))
                  (##f64vector-set! a (##fixnum.+ j 10)
                    (##flonum.- (##flonum.* wk1r x0r)
                                (##flonum.* wk1i x0i)))
                  (##f64vector-set! a (##fixnum.+ j 11)
                    (##flonum.+ (##flonum.* wk1r x0i)
                                (##flonum.* wk1i x0r))))
                (let ((x0r (##flonum.+ x1r x3i))
                      (x0i (##flonum.- x1i x3r)))
                  (##f64vector-set! a (##fixnum.+ j 14)
                    (##flonum.- (##flonum.* wk3r x0r)
                                (##flonum.* wk3i x0i)))
                  (##f64vector-set! a (##fixnum.+ j 15)
                    (##flonum.+ (##flonum.* wk3r x0i)
                                (##flonum.* wk3i x0r)))))))
          (loop (##fixnum.+ j 16) (##fixnum.+ k1 2))))))

  (define (cftmdl n l a w)
    (let ((m (##fixnum.arithmetic-shift-left l 2)))

      (let loop1 ((j 0))
        (##declare (not interrupts-enabled))
        (if (##fixnum.< j l)
          (let* ((j1 (##fixnum.+ j l))
                 (j2 (##fixnum.+ j1 l))
                 (j3 (##fixnum.+ j2 l)))
            (let ((aj00 (##f64vector-ref a (##fixnum.+ j  0)))
                  (aj01 (##f64vector-ref a (##fixnum.+ j  1)))
                  (aj10 (##f64vector-ref a (##fixnum.+ j1 0)))
                  (aj11 (##f64vector-ref a (##fixnum.+ j1 1)))
                  (aj20 (##f64vector-ref a (##fixnum.+ j2 0)))
                  (aj21 (##f64vector-ref a (##fixnum.+ j2 1)))
                  (aj30 (##f64vector-ref a (##fixnum.+ j3 0)))
                  (aj31 (##f64vector-ref a (##fixnum.+ j3 1))))
              (let ((x0r (##flonum.+ aj00 aj10))
                    (x0i (##flonum.+ aj01 aj11))
                    (x1r (##flonum.- aj00 aj10))
                    (x1i (##flonum.- aj01 aj11))
                    (x2r (##flonum.+ aj20 aj30))
                    (x2i (##flonum.+ aj21 aj31))
                    (x3r (##flonum.- aj20 aj30))
                    (x3i (##flonum.- aj21 aj31)))
                (##f64vector-set! a (##fixnum.+ j  0) (##flonum.+ x0r x2r))
                (##f64vector-set! a (##fixnum.+ j  1) (##flonum.+ x0i x2i))
                (##f64vector-set! a (##fixnum.+ j2 0) (##flonum.- x0r x2r))
                (##f64vector-set! a (##fixnum.+ j2 1) (##flonum.- x0i x2i))
                (##f64vector-set! a (##fixnum.+ j1 0) (##flonum.- x1r x3i))
                (##f64vector-set! a (##fixnum.+ j1 1) (##flonum.+ x1i x3r))
                (##f64vector-set! a (##fixnum.+ j3 0) (##flonum.+ x1r x3i))
                (##f64vector-set! a (##fixnum.+ j3 1) (##flonum.- x1i x3r))
                (loop1 (##fixnum.+ j 2)))))))

      (let ((wk1r (##f64vector-ref w 2)))
        (let loop2 ((j m))
          (##declare (not interrupts-enabled))
          (if (##fixnum.< j (##fixnum.+ l m))
            (let* ((j1 (##fixnum.+ j l))
                   (j2 (##fixnum.+ j1 l))
                   (j3 (##fixnum.+ j2 l)))
              (let ((aj00 (##f64vector-ref a (##fixnum.+ j  0)))
                    (aj01 (##f64vector-ref a (##fixnum.+ j  1)))
                    (aj10 (##f64vector-ref a (##fixnum.+ j1 0)))
                    (aj11 (##f64vector-ref a (##fixnum.+ j1 1)))
                    (aj20 (##f64vector-ref a (##fixnum.+ j2 0)))
                    (aj21 (##f64vector-ref a (##fixnum.+ j2 1)))
                    (aj30 (##f64vector-ref a (##fixnum.+ j3 0)))
                    (aj31 (##f64vector-ref a (##fixnum.+ j3 1))))
                (let ((x0r (##flonum.+ aj00 aj10))
                      (x0i (##flonum.+ aj01 aj11))
                      (x1r (##flonum.- aj00 aj10))
                      (x1i (##flonum.- aj01 aj11))
                      (x2r (##flonum.+ aj20 aj30))
                      (x2i (##flonum.+ aj21 aj31))
                      (x3r (##flonum.- aj20 aj30))
                      (x3i (##flonum.- aj21 aj31)))
                  (##f64vector-set! a (##fixnum.+ j  0) (##flonum.+ x0r x2r))
                  (##f64vector-set! a (##fixnum.+ j  1) (##flonum.+ x0i x2i))
                  (##f64vector-set! a (##fixnum.+ j2 0) (##flonum.- x2i x0i))
                  (##f64vector-set! a (##fixnum.+ j2 1) (##flonum.- x0r x2r))
                  (let ((x0r (##flonum.- x1r x3i))
                        (x0i (##flonum.+ x1i x3r)))
                    (##f64vector-set! a (##fixnum.+ j1 0)
                      (##flonum.* wk1r (##flonum.- x0r x0i)))
                    (##f64vector-set! a (##fixnum.+ j1 1)
                      (##flonum.* wk1r (##flonum.+ x0r x0i))))
                  (let ((x0r (##flonum.+ x3i x1r))
                        (x0i (##flonum.- x3r x1i)))
                    (##f64vector-set! a (##fixnum.+ j3 0)
                      (##flonum.* wk1r (##flonum.- x0i x0r)))
                    (##f64vector-set! a (##fixnum.+ j3 1)
                      (##flonum.* wk1r (##flonum.+ x0i x0r))))
                  (loop2 (##fixnum.+ j 2))))))))

      (let ((m2 (##fixnum.arithmetic-shift-left m 1)))
        (let loop3 ((k m2) (k1 2))
          (if (##fixnum.< k n)
            (let* ((k2 (##fixnum.arithmetic-shift-left k1 1))
                   (wk2r (##f64vector-ref w (##fixnum.+ k1 0)))
                   (wk2i (##f64vector-ref w (##fixnum.+ k1 1)))
                   (wk1r (##f64vector-ref w (##fixnum.+ k2 0)))
                   (wk1i (##f64vector-ref w (##fixnum.+ k2 1)))
                   (wk3r
                    (##flonum.- wk1r
                                (##flonum.* (macro-inexact-+2)
                                            (##flonum.* wk2i wk1i))))
                   (wk3i
                    (##flonum.- (##flonum.* (macro-inexact-+2)
                                            (##flonum.* wk2i wk1r))
                                wk1i)))
              (let loop4 ((j k))
                (if (##fixnum.< j (##fixnum.+ l k))

                  (let* ((j1 (##fixnum.+ j l))
                         (j2 (##fixnum.+ j1 l))
                         (j3 (##fixnum.+ j2 l)))
                    (let ((aj00 (##f64vector-ref a (##fixnum.+ j  0)))
                          (aj01 (##f64vector-ref a (##fixnum.+ j  1)))
                          (aj10 (##f64vector-ref a (##fixnum.+ j1 0)))
                          (aj11 (##f64vector-ref a (##fixnum.+ j1 1)))
                          (aj20 (##f64vector-ref a (##fixnum.+ j2 0)))
                          (aj21 (##f64vector-ref a (##fixnum.+ j2 1)))
                          (aj30 (##f64vector-ref a (##fixnum.+ j3 0)))
                          (aj31 (##f64vector-ref a (##fixnum.+ j3 1))))
                      (let ((x0r (##flonum.+ aj00 aj10))
                            (x0i (##flonum.+ aj01 aj11))
                            (x1r (##flonum.- aj00 aj10))
                            (x1i (##flonum.- aj01 aj11))
                            (x2r (##flonum.+ aj20 aj30))
                            (x2i (##flonum.+ aj21 aj31))
                            (x3r (##flonum.- aj20 aj30))
                            (x3i (##flonum.- aj21 aj31)))
                        (##f64vector-set! a (##fixnum.+ j 0)
                          (##flonum.+ x0r x2r))
                        (##f64vector-set! a (##fixnum.+ j 1)
                          (##flonum.+ x0i x2i))
                        (let ((x0r (##flonum.- x0r x2r))
                              (x0i (##flonum.- x0i x2i)))
                          (##f64vector-set! a (##fixnum.+ j2 0)
                            (##flonum.- (##flonum.* wk2r x0r)
                                        (##flonum.* wk2i x0i)))
                          (##f64vector-set! a (##fixnum.+ j2 1)
                            (##flonum.+ (##flonum.* wk2r x0i)
                                        (##flonum.* wk2i x0r))))
                        (let ((x0r (##flonum.- x1r x3i))
                              (x0i (##flonum.+ x1i x3r)))
                          (##f64vector-set! a (##fixnum.+ j1 0)
                            (##flonum.- (##flonum.* wk1r x0r)
                                        (##flonum.* wk1i x0i)))
                          (##f64vector-set! a (##fixnum.+ j1 1)
                            (##flonum.+ (##flonum.* wk1r x0i)
                                        (##flonum.* wk1i x0r))))
                        (let ((x0r (##flonum.+ x1r x3i))
                              (x0i (##flonum.- x1i x3r)))
                          (##f64vector-set! a (##fixnum.+ j3 0)
                            (##flonum.- (##flonum.* wk3r x0r)
                                        (##flonum.* wk3i x0i)))
                          (##f64vector-set! a (##fixnum.+ j3 1)
                            (##flonum.+ (##flonum.* wk3r x0i)
                                        (##flonum.* wk3i x0r))))
                        (loop4 (##fixnum.+ j 2)))))

                  (let* ((wk1r
                          (##f64vector-ref w (##fixnum.+ k2 2)))
                         (wk1i
                          (##f64vector-ref w (##fixnum.+ k2 3)))
                         (wk3r
                          (##flonum.- wk1r
                                      (##flonum.* (macro-inexact-+2)
                                                  (##flonum.* wk2r wk1i))))
                         (wk3i
                          (##flonum.- (##flonum.* (macro-inexact-+2)
                                                  (##flonum.* wk2r wk1r))
                                      wk1i)))

                    (let loop5 ((j (##fixnum.+ k m)))
                      (##declare (not interrupts-enabled))
                      (if (##fixnum.< j (##fixnum.+ l k m))
                        (let* ((j1 (##fixnum.+ j l))
                               (j2 (##fixnum.+ j1 l))
                               (j3 (##fixnum.+ j2 l)))
                          (let ((aj00 (##f64vector-ref a (##fixnum.+ j  0)))
                                (aj01 (##f64vector-ref a (##fixnum.+ j  1)))
                                (aj10 (##f64vector-ref a (##fixnum.+ j1 0)))
                                (aj11 (##f64vector-ref a (##fixnum.+ j1 1)))
                                (aj20 (##f64vector-ref a (##fixnum.+ j2 0)))
                                (aj21 (##f64vector-ref a (##fixnum.+ j2 1)))
                                (aj30 (##f64vector-ref a (##fixnum.+ j3 0)))
                                (aj31 (##f64vector-ref a (##fixnum.+ j3 1))))
                            (let ((x0r (##flonum.+ aj00 aj10))
                                  (x0i (##flonum.+ aj01 aj11))
                                  (x1r (##flonum.- aj00 aj10))
                                  (x1i (##flonum.- aj01 aj11))
                                  (x2r (##flonum.+ aj20 aj30))
                                  (x2i (##flonum.+ aj21 aj31))
                                  (x3r (##flonum.- aj20 aj30))
                                  (x3i (##flonum.- aj21 aj31)))
                              (##f64vector-set! a (##fixnum.+ j 0)
                                (##flonum.+ x0r x2r))
                              (##f64vector-set! a (##fixnum.+ j 1)
                                (##flonum.+ x0i x2i))
                              (let ((x0r (##flonum.- x0r x2r))
                                    (x0i (##flonum.- x0i x2i)))
                                (##f64vector-set! a (##fixnum.+ j2 0)
                                  (##flonum.-
                                   (##flonum.+ (##flonum.* wk2i x0r)
                                               (##flonum.* wk2r x0i))))
                                (##f64vector-set! a (##fixnum.+ j2 1)
                                  (##flonum.- (##flonum.* wk2r x0r)
                                              (##flonum.* wk2i x0i))))
                              (let ((x0r (##flonum.- x1r x3i))
                                    (x0i (##flonum.+ x1i x3r)))
                                (##f64vector-set! a (##fixnum.+ j1 0)
                                  (##flonum.- (##flonum.* wk1r x0r)
                                              (##flonum.* wk1i x0i)))
                                (##f64vector-set! a (##fixnum.+ j1 1)
                                  (##flonum.+ (##flonum.* wk1r x0i)
                                              (##flonum.* wk1i x0r))))
                              (let ((x0r (##flonum.+ x1r x3i))
                                    (x0i (##flonum.- x1i x3r)))
                                (##f64vector-set! a (##fixnum.+ j3 0)
                                  (##flonum.- (##flonum.* wk3r x0r)
                                              (##flonum.* wk3i x0i)))
                                (##f64vector-set! a (##fixnum.+ j3 1)
                                  (##flonum.+ (##flonum.* wk3r x0i)
                                              (##flonum.* wk3i x0r))))
                              (loop5 (##fixnum.+ j 2)))))))

                    (loop3 (##fixnum.+ k m2) (##fixnum.+ k1 2)))))))))))

  (define (rftfsub n a nc c)
    (let ((ks (##fixnum.quotient (##fixnum.arithmetic-shift-left nc 2) n))
          (m (##fixnum.arithmetic-shift-right n 1)))
      (let loop ((k 2)
                 (kk ks))
        (##declare (not interrupts-enabled))
        (if (##fixnum.< k m)
          (let ((j (##fixnum.- n k)))
            (let ((wkr
                   (##flonum.- (macro-inexact-+1/2)
                               (##f64vector-ref c (##fixnum.- nc kk))))
                  (wki (##f64vector-ref c kk))
                  (aj0 (##f64vector-ref a (##fixnum.+ j 0)))
                  (aj1 (##f64vector-ref a (##fixnum.+ j 1)))
                  (ak0 (##f64vector-ref a (##fixnum.+ k 0)))
                  (ak1 (##f64vector-ref a (##fixnum.+ k 1))))
              (let ((xr (##flonum.- ak0 aj0))
                    (xi (##flonum.+ ak1 aj1)))
                (let ((yr
                       (##flonum.- (##flonum.* wkr xr) (##flonum.* wki xi)))
                      (yi
                       (##flonum.+ (##flonum.* wkr xi) (##flonum.* wki xr))))
                  (##f64vector-set! a (##fixnum.+ k 0) (##flonum.- ak0 yr))
                  (##f64vector-set! a (##fixnum.+ k 1) (##flonum.- ak1 yi))
                  (##f64vector-set! a (##fixnum.+ j 0) (##flonum.+ aj0 yr))
                  (##f64vector-set! a (##fixnum.+ j 1) (##flonum.- aj1 yi)))))
            (loop (##fixnum.+ k 2)
                  (##fixnum.+ kk ks)))))))

  (define (rftbsub n a nc c)
    (let ((ks (##fixnum.quotient (##fixnum.arithmetic-shift-left nc 2) n))
          (m (##fixnum.arithmetic-shift-right n 1)))
      (##f64vector-set! a 1 (##flonum.- (##f64vector-ref a 1)))
      (let loop ((k 2)
                 (kk ks))
        (##declare (not interrupts-enabled))
        (if (##fixnum.< k m)
          (let ((j (##fixnum.- n k)))
            (let ((wkr
                   (##flonum.- (macro-inexact-+1/2)
                               (##f64vector-ref c (##fixnum.- nc kk))))
                  (wki (##f64vector-ref c kk))
                  (aj0 (##f64vector-ref a (##fixnum.+ j 0)))
                  (aj1 (##f64vector-ref a (##fixnum.+ j 1)))
                  (ak0 (##f64vector-ref a (##fixnum.+ k 0)))
                  (ak1 (##f64vector-ref a (##fixnum.+ k 1))))
              (let ((xr (##flonum.- ak0 aj0))
                    (xi (##flonum.+ ak1 aj1)))
                (let ((yr
                       (##flonum.+ (##flonum.* wkr xr) (##flonum.* wki xi)))
                      (yi
                       (##flonum.- (##flonum.* wkr xi) (##flonum.* wki xr))))
                  (##f64vector-set! a (##fixnum.+ k 0) (##flonum.- ak0 yr))
                  (##f64vector-set! a (##fixnum.+ k 1) (##flonum.- yi ak1))
                  (##f64vector-set! a (##fixnum.+ j 0) (##flonum.+ aj0 yr))
                  (##f64vector-set! a (##fixnum.+ j 1) (##flonum.- yi aj1)))))
            (loop (##fixnum.+ k 2)
                  (##fixnum.+ kk ks)))
          (##f64vector-set! a (##fixnum.+ m 1)
            (##flonum.- (##f64vector-ref a (##fixnum.+ m 1))))))))

  (define (rdft n isgn a ip ipcontrol w wc)
    (let* ((nw0
            (##vector-ref ipcontrol 0))
           (nw
            (if (##fixnum.< (##fixnum.arithmetic-shift-left nw0 2) n)
              (let ((nw (##fixnum.arithmetic-shift-right n 2)))
                (makewt nw ip ipcontrol w)
                nw)
              nw0))
           (nc0
            (##vector-ref ipcontrol 1))
           (nc
            (if (##fixnum.< (##fixnum.arithmetic-shift-left nc0 2) n)
              (let ((nc (##fixnum.arithmetic-shift-right n 2)))
                (makect nc ipcontrol wc)
                nc)
              nc0)))

      (if (##fixnum.< isgn 0)
        (begin
          (##f64vector-set! a 1
            (##flonum.* (macro-inexact-+1/2)
                        (##flonum.- (##f64vector-ref a 0)
                                    (##f64vector-ref a 1))))
          (##f64vector-set! a 0
            (##flonum.- (##f64vector-ref a 0)
                        (##f64vector-ref a 1)))
          (rftbsub n a nc wc)
          (bitrv2 n ip a)
          (cftbsub n a w))
        (begin
          (bitrv2 n ip a)
          (cftfsub n a w)
          (rftfsub n a nc wc)
          (let ((xi (##flonum.- (##f64vector-ref a 0) (##f64vector-ref a 1))))
            (##f64vector-set! a 0
              (##flonum.+ (##f64vector-ref a 0)
                          (##f64vector-ref a 1)))
            (##f64vector-set! a 1 xi))))))

  (define (fft-mul x y)

    (define (make-fft-state ipcontrol w wc ip a b)
      (##vector ipcontrol w wc ip a b))

    (define (fft-state-ipcontrol state)
      (##vector-ref state 0))

    (define (fft-state-w state)
      (##vector-ref state 1))

    (define (fft-state-wc state)
      (##vector-ref state 2))

    (define (fft-state-ip state)
      (##vector-ref state 3))

    (define (fft-state-a state)
      (##vector-ref state 4))

    (define (fft-state-b state)
      (##vector-ref state 5))

    (define (make-new-state two-n)
      (make-fft-state
       (##vector 0 0)
       (##make-f64vector (##fixnum.arithmetic-shift-right two-n 2)
                         (macro-inexact-+0))
       (##make-f64vector (##fixnum.arithmetic-shift-right two-n 2)
                         (macro-inexact-+0))
       (##make-vector (##fixnum.+
                       2
                       (##flonum.->fixnum
                        (##flonum.sqrt
                         (##flonum.<-fixnum two-n))))
                      0)
       (##make-f64vector two-n (macro-inexact-+0))
       (##make-f64vector two-n (macro-inexact-+0))))

    (let* ((x-length
            (##bignum.fdigit-length x))
           (y-length
            (##bignum.fdigit-length y))
           (mn
            (##fixnum.max x-length y-length))
           (two-n
            (##fixnum.arithmetic-shift-left
             2
             (##integer-length (##fixnum.- mn 1))))
           (normalizer
            (##flonum./
             (##flonum.<-fixnum (##fixnum.arithmetic-shift-right two-n 1))))
           (internal-state
            (make-new-state two-n)))

      (let ((ipcontrol (fft-state-ipcontrol internal-state))
            (w         (fft-state-w         internal-state))
            (wc        (fft-state-wc        internal-state))
            (ip        (fft-state-ip        internal-state))
            (a         (fft-state-a         internal-state)))

        (let loop1 ((i 0))
          (##declare (not interrupts-enabled))
          (if (##fixnum.< i x-length)
            (begin
              (##f64vector-set! a i
                (##flonum.<-fixnum (##bignum.fdigit-ref x i)))
              (loop1 (##fixnum.+ i 1)))))

        (rdft two-n 1 a ip ipcontrol w wc)

        (if (##eq? x y) ; fast handling of squaring an integer

          (begin
            (##f64vector-set! a 0
              (##flonum.* (##f64vector-ref a 0) (##f64vector-ref a 0)))
            (##f64vector-set! a 1
              (##flonum.* (##f64vector-ref a 1) (##f64vector-ref a 1)))
            (let loop2 ((i 2))
              (##declare (not interrupts-enabled))
              (if (##fixnum.< i two-n)
                (let ((ai (##f64vector-ref a i))
                      (ai+1 (##f64vector-ref a (##fixnum.+ i 1))))
                  (##f64vector-set! a i
                    (##flonum.- (##flonum.* ai ai)   (##flonum.* ai+1 ai+1)))
                  (##f64vector-set! a (##fixnum.+ i 1)
                    (##flonum.+ (##flonum.* ai ai+1) (##flonum.* ai   ai+1)))
                  (loop2 (##fixnum.+ i 2))))))

            (let ((b (fft-state-b internal-state)))
              (let loop3 ((i 0))
                (##declare (not interrupts-enabled))
                (if (##fixnum.< i y-length)
                  (begin
                    (##f64vector-set! b i
                      (##flonum.<-fixnum (##bignum.fdigit-ref y i)))
                    (loop3 (##fixnum.+ i 1)))))
              (rdft two-n 1 b ip ipcontrol w wc)
              (##f64vector-set! a 0
                (##flonum.* (##f64vector-ref b 0) (##f64vector-ref a 0)))
              (##f64vector-set! a 1
                (##flonum.* (##f64vector-ref b 1) (##f64vector-ref a 1)))
              (let loop4 ((i 2))
                (##declare (not interrupts-enabled))
                (if (##fixnum.< i two-n)
                  (let ((ai (##f64vector-ref a i))
                        (ai+1 (##f64vector-ref a (##fixnum.+ i 1)))
                        (bi (##f64vector-ref b i))
                        (bi+1 (##f64vector-ref b (##fixnum.+ i 1))))
                    (##f64vector-set! a i
                      (##flonum.- (##flonum.* bi ai)   (##flonum.* ai+1 bi+1)))
                    (##f64vector-set! a (##fixnum.+ i 1)
                      (##flonum.+ (##flonum.* bi ai+1) (##flonum.* ai   bi+1)))
                    (loop4 (##fixnum.+ i 2)))))))

        (rdft two-n -1 a ip ipcontrol w wc)

        (let* ((result-length
                (##fixnum.+ x-length y-length))
               (fbase
                (##flonum.<-fixnum ##bignum.fdigit-base))
               (result
                (##bignum.make
                 (##fixnum.quotient
                  result-length
                  (##fixnum.quotient ##bignum.adigit-width
                                     ##bignum.fdigit-width))
                 #f
                 #f)))
          (let ((loop-carry (##f64vector (macro-inexact-+0))))
            (let loop5 ((i 0))
              (##declare (not interrupts-enabled))
              (if (##fixnum.< i result-length)
                (let* ((t
                        (##flonum.+ (##flonum.+ (##f64vector-ref loop-carry 0)
                                                (macro-inexact-+1/2))
                                    (##flonum.* (##f64vector-ref a i)
                                                normalizer)))
                       (carry
                        (##flonum.floor (##flonum./ t fbase)))
                       (digit
                        (##flonum.- t (##flonum.* carry fbase))))
                  (##bignum.fdigit-set! result i (##flonum.->fixnum digit))
                  (##f64vector-set! loop-carry 0 carry)
                  (loop5 (##fixnum.+ i 1)))
                (cleanup x y result))))))))

  (define (naive-mul x x-length y y-length)  ;; multiplies x by each digit of y
    (let ((result
           (##bignum.make
            (##fixnum.+ (##bignum.adigit-length x) (##bignum.adigit-length y))
            #f
            #f)))
      (##declare (not interrupts-enabled))
      (if (##eq? x y)

        (let loop1 ((k 1))   ; multiply off-diagonals
          (if (##fixnum.< k x-length)
            (let ((multiplier (##bignum.mdigit-ref x k)))
              (if (##eq? multiplier 0)
                (loop1 (##fixnum.+ k 1))
                (let loop2 ((i 0)
                            (j k)
                            (carry 0))
                  (if (##fixnum.< i k)
                    (loop2 (##fixnum.+ i 1)
                           (##fixnum.+ j 1)
                           (##bignum.mdigit-mul! result
                                                 j
                                                 x
                                                 i
                                                 multiplier
                                                 carry))
                    (begin
                      (##bignum.mdigit-set! result j carry)
                      (loop1 (##fixnum.+ k 1)))))))
            (let ((result-length (##bignum.adigit-length result)))
              (let loop3 ((k 0)       ; double the off-diagonal terms
                          (carry 0))
                (if (##fixnum.< k result-length)
                  (loop3 (##fixnum.+ k 1)
                         (##bignum.adigit-add! result
                                               k
                                               result
                                               k
                                               carry))
                  (let ((shift ##bignum.mdigit-width)
                        (mask ##bignum.mdigit-base-minus-1))
                    (let loop4 ((k 0)              ; add squares of diagonals
                                (two-k 0)
                                (carry 0))
                      (if (##fixnum.< k x-length)
                        (let ((next-digit
                               (##fixnum.+ (##bignum.mdigit-mul!
                                            result
                                            two-k
                                            x
                                            k
                                            (##bignum.mdigit-ref x k)
                                            carry)
                                           (##bignum.mdigit-ref
                                            result
                                            (##fixnum.+ two-k 1)))))
                          (##bignum.mdigit-set!
                           result
                           (##fixnum.+ two-k 1)
                           (##fixnum.bitwise-and next-digit mask))
                          (loop4 (##fixnum.+ k 1)
                                 (##fixnum.+ two-k 2)
                                 (##fixnum.arithmetic-shift-right
                                  next-digit
                                  shift)))
                        (cleanup x y result)))))))))

        (let loop1 ((k 0))
          (##declare (not interrupts-enabled))
          (if (##fixnum.< k y-length)
            (let ((multiplier (##bignum.mdigit-ref y k)))
              (if (##eq? multiplier 0)
                (loop1 (##fixnum.+ k 1))
                (let loop2 ((i 0)
                            (j k)
                            (carry 0))
                  (if (##fixnum.< i x-length)
                    (loop2 (##fixnum.+ i 2)
                           (##fixnum.+ j 2)
                           (##bignum.mdigit-mul!
                            result
                            (##fixnum.+ j 1)
                            x
                            (##fixnum.+ i 1)
                            multiplier
                            (##bignum.mdigit-mul! result
                                                  j
                                                  x
                                                  i
                                                  multiplier
                                                  carry)))
                    (begin
                      (##bignum.mdigit-set! result j carry)
                      (loop1 (##fixnum.+ k 1)))))))
            (cleanup x y result))))))

  (define (cleanup x y result)

    ; Both naive-mul and fft-mul do unsigned multiplies, fix that here.

    (define (fix x y result)

      (##declare (not interrupts-enabled))
      ;; In theory interrupts can be disabled here.  However, the C
      ;; code generated in this case is not compiled properly by
      ;; gcc (egcs-2.91.66), so the above declaration is commented out.

      ;; Look, *nobody* uses egcs-2.91.66 anymore, so let's disable
      ;; interrupts, OK?  It makes a relatively big difference for small
      ;; bignums.

      (if (##bignum.negative? y)
        (let ((x-length (##bignum.adigit-length x)))
          (let loop ((i 0)
                     (j (##bignum.adigit-length y))
                     (borrow 0))
            (if (##fixnum.< i x-length)
              (loop (##fixnum.+ i 1)
                    (##fixnum.+ j 1)
                    (##bignum.adigit-sub! result j x i borrow)))))))

    (fix x y result)
    (fix y x result)
    (##bignum.normalize! result))

  (define (karatsuba-mul x y)
    (let* ((x-length
            (##bignum.adigit-length x))
           (y-length
            (##bignum.adigit-length y))
           (shift-digits
            (##fixnum.arithmetic-shift-right y-length 1))
           (shift-bits
            (##fixnum.* shift-digits ##bignum.adigit-width))
           (y-high
            (##bignum.arithmetic-shift y (##fixnum.- shift-bits)))
           (y-low
            (##extract-bit-field shift-bits 0 y)))
      (if (##eq? x y)
          (let ((high-term
                 (##* y-high y-high))
                (low-term
                 (##* y-low y-low))
                (mid-term
                 (let ((arg (##- y-high y-low)))
                   (##* arg arg))))
            (##+ (##arithmetic-shift high-term (##fixnum.* shift-bits 2))
                 (##+ (##arithmetic-shift
                       (##+ high-term
                            (##- low-term mid-term))
                       shift-bits)
                      low-term)))
          (let ((x-high
                 (##bignum.arithmetic-shift x (##fixnum.- shift-bits)))
                (x-low
                 (##extract-bit-field shift-bits 0 x)))
            (let ((high-term
                   (##* x-high y-high))
                  (low-term
                   (##* x-low y-low))
                  (mid-term
                   (##* (##- x-high x-low)
                        (##- y-high y-low))))
              (##+ (##arithmetic-shift high-term (##fixnum.* shift-bits 2))
                   (##+ (##arithmetic-shift
                         (##+ high-term
                              (##- low-term mid-term))
                         shift-bits)
                        low-term)))))))

  (define (mul x x-length y y-length) ; x-length <= y-length
    (let ((x-width (##fixnum.* x-length ##bignum.mdigit-width)))
      (cond ((##fixnum.< x-width ##bignum.naive-mul-max-width)
             (naive-mul y y-length x x-length))
            ((##fixnum.> y-length (##fixnum.+ x-length x-length))
             (let ((y-width/2 (##fixnum.* (##fixnum.arithmetic-shift-right
                                           (##bignum.adigit-length y)
                                           1)
                                          ##bignum.adigit-width)))
               (##+ (##arithmetic-shift
                     (##* x (##arithmetic-shift y (##fixnum.- y-width/2)))
                     y-width/2)
                    (##* x (##extract-bit-field y-width/2 0 y)))))
            ((or (##fixnum.< x-width ##bignum.fft-mul-min-width)
                 (##fixnum.< ##bignum.fft-mul-max-width
                             (##fixnum.* y-length ##bignum.mdigit-width)))
             (karatsuba-mul x y))
            (else
             (fft-mul x y)))))

  ;; Certain decisions must be made for multiplication.
  ;; First, if both bignums are small, just do naive mul to avoid further overhead.
  ;; This is done in the main body of ##bignum.*.
  ;; Second, if it would help to shift out low-order zeros of an argument, do so.  That's
  ;; done in the main body of ##bignum.*
  ;; Third, if one number is more than twice the length of the other and you're
  ;; doing karatsuba or fft, it helps to break the larger number in two.  This is
  ;; done in mul.
  ;; Finally, one must decide whether one is using naive mul, karatsuba, or fft.
  ;; This is done in mul.

  (define (low-bits-to-shift x)
    (let ((size (##integer-length x))
          (low-bits (##first-set-bit x)))
      (if (##fixnum.< size (##fixnum.+ low-bits low-bits))
          low-bits
          0)))

  (define (possibly-unnormalized-bignum-arithmetic-shift x bits)
    (if (##eq? bits 0)
        (if (##fixnum.= (##bignum.adigit-length x) 1)
            (##bignum.normalize! x)
            x)
        (##arithmetic-shift x bits)))

  (let ((x-length (##bignum.mdigit-length x))
        (y-length (##bignum.mdigit-length y)))
    (cond ((or (##not (use-fast-bignum-algorithms))
               (and (##fixnum.< x-length 50)
                    (##fixnum.< y-length 50)))
           (if (##fixnum.< x-length y-length)
               (naive-mul y y-length x x-length)
               (naive-mul x x-length y y-length)))
          ((##eq? x y)
           (let ((low-bits (low-bits-to-shift x)))
             (if (##eq? low-bits 0)
                 (mul x x-length x x-length)
                 (##arithmetic-shift
                  (##exact-int.square (##arithmetic-shift x (##fixnum.- low-bits)))
                  (##fixnum.+ low-bits low-bits)))))
          (else
           (let ((x-low-bits (low-bits-to-shift x))
                 (y-low-bits (low-bits-to-shift y)))
             (if (##eq? (##fixnum.+ x-low-bits y-low-bits) 0)
                 (if (##fixnum.< x-length y-length)
                     (mul x x-length y y-length)
                     (mul y y-length x x-length))
                 (##arithmetic-shift
                  (##* (possibly-unnormalized-bignum-arithmetic-shift x (##fixnum.- x-low-bits))
                       (possibly-unnormalized-bignum-arithmetic-shift y (##fixnum.- y-low-bits)))
                  (##fixnum.+ x-low-bits y-low-bits))))))))

(define-prim (##bignum.arithmetic-shift x shift)
  (let* ((digit-shift
          (if (##fixnum.< shift 0)
            (##fixnum.- (##fixnum.quotient (##fixnum.+ shift 1)
                                           ##bignum.adigit-width)
                        1)
            (##fixnum.quotient shift ##bignum.adigit-width)))
         (bit-shift
          (##fixnum.modulo shift ##bignum.adigit-width))
         (x-length
          (##bignum.adigit-length x))
         (result-length
          (##fixnum.+ (##fixnum.+ x-length digit-shift)
                      (if (##fixnum.zero? bit-shift) 0 1))))
    (if (##fixnum.< 0 result-length)

      (let ((result (##bignum.make result-length #f #f)))

        (##declare (not interrupts-enabled))

        (if (##fixnum.zero? bit-shift)
          (let ((smallest-i (##fixnum.max 0 digit-shift)))
            (let loop1 ((i (##fixnum.- result-length 1))
                        (j (##fixnum.- x-length 1)))
              (if (##fixnum.< i smallest-i)
                (##bignum.normalize! result)
                (begin
                  (##bignum.adigit-copy! result i x j)
                  (loop1 (##fixnum.- i 1)
                         (##fixnum.- j 1))))))
          (let ((left-fill (if (##bignum.negative? x)
                             ##bignum.adigit-ones
                             ##bignum.adigit-zeros))
                (i (##fixnum.- result-length 1))
                (j (##fixnum.- x-length 1))
                (divider bit-shift)
                (smallest-i (##fixnum.max 0 (##fixnum.+ digit-shift 1))))
            (##bignum.adigit-cat! result i left-fill 0 x j divider)
            (let loop2 ((i (##fixnum.- i 1))
                        (j (##fixnum.- j 1)))
              (if (##fixnum.< i smallest-i)
                (begin
                  (if (##not (##fixnum.< i 0))
                    (##bignum.adigit-cat! result
                                          i
                                          x
                                          (##fixnum.+ j 1)
                                          ##bignum.adigit-zeros
                                          0
                                          divider))
                  (##bignum.normalize! result))
                (begin
                  (##bignum.adigit-cat! result
                                        i
                                        x
                                        (##fixnum.+ j 1)
                                        x
                                        j
                                        divider)
                  (loop2 (##fixnum.- i 1)
                         (##fixnum.- j 1))))))))

      (if (##bignum.negative? x)
        -1
        0))))

; Bignum division.

(define ##reciprocal-cache (##make-table 0 #f #t #f ##eq?))

(define ##bignum.mdigit-width/2
  (##fixnum.quotient ##bignum.mdigit-width 2))

(define ##bignum.mdigit-base*16
  (##fixnum.* ##bignum.mdigit-base 16))

(define-prim (##bignum.div u v)

  ; u is an unnormalized bignum, v is a normalized exact-int
  ; 0 < v <= u

  (define (##exact-int.reciprocal v bits)

    ;; returns an approximation to the reciprocal of
    ;; .v1 v2 v3 ...
    ;; where v1 is the highest set bit of v; result is of the form
    ;; xx . xxxxxxxxxxxxxxxxxxx where there are bits + 1 bits to the
    ;; right of the binary point. The result is always <= 2; see Knuth, volume 2.

    (let ((cached-value (##table-ref ##reciprocal-cache v #f)))
      (if (and cached-value
               (##not (##fixnum.< (##cdr cached-value) bits)))
          cached-value
          (let ((v-length (##integer-length v)))

            (define (recip v bits)
              (cond ((and cached-value
                          (##not (##fixnum.< (##cdr cached-value) bits)))
                     cached-value)
                    ((##fixnum.<= bits ##bignum.mdigit-width/2)
                     (##cons (##fixnum.quotient
                              ##bignum.mdigit-base*16
                              (##arithmetic-shift
                               v
                               (##fixnum.- ##bignum.mdigit-width/2 -3 v-length)))
                             ##bignum.mdigit-width/2))
                    (else
                     (let* ((high-bits
                             (##fixnum.arithmetic-shift-right
                              (##fixnum.+ bits 1)
                              1))
                            (z-bits      ; >= high-bits + 1 to right of point
                             (recip v high-bits))
                            (z           ; high-bits + 1 to right of point
                             (##arithmetic-shift
                              (##car z-bits)
                              (##fixnum.- high-bits (##cdr z-bits))))
                            (v-bits      ; bits + 3 to right of point
                             (##arithmetic-shift
                              v
                              (##fixnum.- (##fixnum.+ bits 3)
                                          v-length)))
                            (v*z*z       ; 2 * high-bits + bits + 5 to right
                             (##* v-bits (##exact-int.square z)))
                            (two-z       ; 2 * high-bits + bits + 5 to right
                             (##arithmetic-shift
                              z
                              (##fixnum.+ high-bits (##fixnum.+ bits 5))))
                            (temp
                             (##- two-z v*z*z))
                            (bits-to-shift
                             (##fixnum.+ 4 (##fixnum.+ high-bits high-bits)))
                            (shifted-temp
                             (##arithmetic-shift
                              temp
                              (##fixnum.- bits-to-shift))))
                       (if (##fixnum.< (##first-set-bit temp) bits-to-shift)
                           (##cons (##+ shifted-temp 1) bits)
                           (##cons shifted-temp bits))))))

            (let ((result (recip v bits)))
              (##table-set! ##reciprocal-cache v result)
              result)))))

  (define (naive-div u v)

    ; u is a normalized bignum, v is an unnormalized bignum
    ; v >= ##bignum.mdigit-base

    (let ((n
           (let loop1 ((i (##fixnum.- (##bignum.mdigit-length v) 1)))
             (if (##fixnum.< 0 (##bignum.mdigit-ref v i))
               (##fixnum.+ i 1)
               (loop1 (##fixnum.- i 1))))))
      (let ((normalizing-bit-shift
             (##fixnum.- ##bignum.mdigit-width
                         (##integer-length
                          (##bignum.mdigit-ref v (##fixnum.- n 1))))))
        (let ((u (##bignum.arithmetic-shift u normalizing-bit-shift))
              (v (##bignum.arithmetic-shift v normalizing-bit-shift)))
          (let ((q
                 (##bignum.make
                  (##fixnum.+ (##fixnum.- (##bignum.adigit-length u)
                                          (##bignum.adigit-length v))
                              2) ; 1 is not always sufficient...
                  #f
                  #f))
                (v_n-1
                 (##bignum.mdigit-ref v (##fixnum.- n 1)))
                (v_n-2 (##bignum.mdigit-ref v (##fixnum.- n 2))))
            (let ((m
                   (let loop2 ((i (##fixnum.- (##bignum.mdigit-length u) 1)))
                     (let ((u_i (##bignum.mdigit-ref u i)))
                       (if (##fixnum.< 0 u_i)
                         (if (##not (##fixnum.< u_i v_n-1))
                           (##fixnum.- (##fixnum.+ i 1) n)
                           (##fixnum.- i n))
                         (loop2 (##fixnum.- i 1)))))))
              (let loop3 ((j m))
                (if (##not (##fixnum.< j 0))
                  (let ((q-hat
                         (let ((q-hat
                                (##bignum.mdigit-quotient
                                 u
                                 (##fixnum.+ j n)
                                 v_n-1))
                               (u_n+j-2
                                (##bignum.mdigit-ref
                                 u
                                 (##fixnum.+ (##fixnum.- j 2) n)
                                 )))
                           (let ((r-hat
                                  (##bignum.mdigit-remainder
                                   u
                                   (##fixnum.+ j n)
                                   v_n-1
                                   q-hat)))
                             (if (or (##fixnum.= q-hat ##bignum.mdigit-base)
                                     (##bignum.mdigit-test?
                                      q-hat
                                      v_n-2
                                      r-hat
                                      u_n+j-2))
                               (let ((q-hat (##fixnum.- q-hat 1))
                                     (r-hat (##fixnum.+ r-hat v_n-1)))
                                 (if (and (##fixnum.< r-hat
                                                      ##bignum.mdigit-base)
                                          (##bignum.mdigit-test?
                                           q-hat
                                           v_n-2
                                           r-hat
                                           u_n+j-2))
                                   (##fixnum.- q-hat 1)
                                   q-hat))
                               q-hat)))))

                    (##declare (not interrupts-enabled))

                    (let loop4 ((i j)
                                (k 0)
                                (borrow 0))
                      (if (##fixnum.< k n)
                        (loop4 (##fixnum.+ i 2)
                               (##fixnum.+ k 2)
                               (##bignum.mdigit-div!
                                u
                                (##fixnum.+ i 1)
                                v
                                (##fixnum.+ k 1)
                                q-hat
                                (##bignum.mdigit-div!
                                 u
                                 i
                                 v
                                 k
                                 q-hat
                                 borrow)))
                        (let ((borrow
                               (if (##fixnum.< n k)
                                 borrow
                                 (##bignum.mdigit-div!
                                  u
                                  i
                                  v
                                  k
                                  q-hat
                                  borrow))))
                          (if (##fixnum.< borrow 0)
                            (let loop5 ((i j)
                                        (l 0)
                                        (carry 0))
                              (if (##fixnum.< n l)
                                (begin
                                  (##bignum.mdigit-set!
                                   q
                                   j
                                   (##fixnum.- q-hat 1))
                                  (loop3 (##fixnum.- j 1)))
                                (loop5 (##fixnum.+ i 1)
                                       (##fixnum.+ l 1)
                                       (##bignum.mdigit-mul!
                                        u
                                        i
                                        v
                                        l
                                        1
                                        carry))))
                            (begin
                              (##bignum.mdigit-set! q j q-hat)
                              (loop3 (##fixnum.- j 1))))))))
                  (##cons (##bignum.normalize! q)
                          (##arithmetic-shift
                           (##bignum.normalize! u)
                           (##fixnum.- normalizing-bit-shift)))))))))))

  (define (div-one u v)
    (let ((m
           (let loop6 ((i (##fixnum.- (##bignum.mdigit-length u) 1)))
             (if (##fixnum.< 0 (##bignum.mdigit-ref u i))
               (##fixnum.+ i 1)
               (loop6 (##fixnum.- i 1))))))
      (let ((work-u (##bignum.make 1 #f #f))
            (q (##bignum.make (##bignum.adigit-length u) #f #f)))

        (##declare (not interrupts-enabled))

        (let loop7 ((i m)
                    (r-hat 0))
          (##bignum.mdigit-set!
           work-u
           1
           r-hat)
          (##bignum.mdigit-set!
           work-u
           0
           (##bignum.mdigit-ref u (##fixnum.- i 1)))
          (let ((q-hat (##bignum.mdigit-quotient work-u 1 v)))
            (let ((r-hat (##bignum.mdigit-remainder work-u 1 v q-hat)))
              (##bignum.mdigit-set! q (##fixnum.- i 1) q-hat)
              (if (##fixnum.< 1 i)
                (loop7 (##fixnum.- i 1)
                       r-hat)
                (let ()
                  (##declare (interrupts-enabled))
                  (##cons (##bignum.normalize! q)
                          r-hat)))))))))

  (define (small-quotient-or-divisor-divide u v)
    ;; Here we do a quick check to catch most cases where the quotient will
    ;; be 1 and do a subtraction.  This comes up a lot in gcd calculations.
    ;; Otherwise, we just call naive-div.
    (let ((u-mlength (##bignum.mdigit-length u))
          (v-mlength (##bignum.mdigit-length v)))
      (if (and (##fixnum.= u-mlength v-mlength)
               (let loop ((i (##fixnum.- u-mlength 1)))
                 (let ((udigit (##bignum.mdigit-ref u i)))
                   (if (##eq? udigit 0)
                     (loop (##fixnum.- i 1))
                     (##fixnum.< udigit
                                 (##fixnum.arithmetic-shift-left
                                  (##bignum.mdigit-ref v i)
                                  1))))))
        (##cons 1 (##- u v))
        (naive-div u v))))

  (define (big-divide u v)

    ; u and v are positive bignums

    (let ((v-length (##integer-length v))
          (v-first-set-bit (##first-set-bit v)))
      ;; first we check whether it may be beneficial to shift out
      ;; low-order zero bits of v
      (if (##fixnum.>= v-first-set-bit
                       (##fixnum.arithmetic-shift-right v-length 1))
        (let ((reduced-quotient
               (##exact-int.div
                (##bignum.arithmetic-shift u (##fixnum.- v-first-set-bit))
                (##bignum.arithmetic-shift v (##fixnum.- v-first-set-bit))))
              (extra-remainder
               (##extract-bit-field v-first-set-bit 0 u)))
            (##cons (##car reduced-quotient)
                    (##+ (##arithmetic-shift (##cdr reduced-quotient)
                                             v-first-set-bit)
                         extra-remainder)))
        (if (##fixnum.< v-length ##bignum.fft-mul-min-width)
          (small-quotient-or-divisor-divide u v)
          (let* ((u-length (##integer-length u))
                 (length-difference (##fixnum.- u-length v-length)))
            (if (##fixnum.< length-difference ##bignum.fft-mul-min-width)
              (small-quotient-or-divisor-divide u v)
              (let* ((z-bits (##exact-int.reciprocal v length-difference))
                     (z (##car z-bits))
                     (bits (##cdr z-bits)))
                (let ((test-quotient
                       (##bignum.arithmetic-shift
                        (##* (##bignum.arithmetic-shift
                              u
                              (##fixnum.- length-difference
                                          (##fixnum.- u-length 2)))
                             (##bignum.arithmetic-shift
                              z
                              (##fixnum.- length-difference bits)))
                        (##fixnum.- -3 length-difference))))
                  (let ((rem (##- u (##* test-quotient v))))
                    ;; I believe, and I haven't found any counterexamples in my tests
                    ;; to disprove it, that test-quotient can be off by at most +-1.
                    ;; I can't prove this, however, so we put in the following loops.

                    ;; Especially note that our reciprocal does not satisfy the
                    ;; error bounds in Knuth's volume 2 in perhaps a vain effort to
                    ;; save some computations. perhaps this should be fixed.  blah.

                    (cond ((##negative? rem)
                           (let loop ((test-quotient test-quotient)
                                      (rem rem))
                             (let ((test-quotient (##- test-quotient 1))
                                   (rem (##+ rem v)))
                               (if (##negative? rem)
                                   (loop test-quotient rem)
                                   (##cons test-quotient rem)))))
                          ((##< rem v)
                           (##cons test-quotient
                                   rem))
                          (else
                           (let loop ((test-quotient test-quotient)
                                      (rem rem))
                             (let ((test-quotient (##+ test-quotient 1))
                                   (rem (##- rem v)))
                               (if (##< rem v)
                                   (##cons test-quotient rem)
                                   (loop test-quotient rem)))))))))))))))

  (if (##fixnum? v)
    (if (##fixnum.< v ##bignum.mdigit-base)
      (div-one u v)
      (begin
        ;; here it's probably not worth the extra cycles to check whether
        ;; a subtraction would be sufficient, i.e., we don't call
        ;; short-divisor-or-quotient-divide
        (naive-div u (##bignum.<-fixnum v))))
    (if (use-fast-bignum-algorithms)
        (big-divide u v)
        (naive-div u v))))

;------------------------------------------------------------------------------

; Exact integer operations
; ------------------------

(define-prim (##exact-int.*-expt2 x y)
  (if (##fixnum.negative? y)
    (##ratnum.normalize x (##arithmetic-shift 1 (##fixnum.- y)))
    (##arithmetic-shift x y)))

(define-prim (##exact-int.div x y)
  (cond ((##eq? y 1)
         (##cons x 0))
        ((##eq? y -1)
         (##cons (##negate x) 0))
        ((##eq? x y)              ;; can come up in rational arithmetic
         (##cons 1 0))
        (else
         (if (##fixnum? x)
           (if (##fixnum? y)
             (##cons (##fixnum.quotient x y) ; note: y can't be -1
                     (##fixnum.remainder x y))
             (##cons 0 x))
           (let* ((x-negative?
                   (##bignum.negative? x))
                  (abs-x
                   (if x-negative? (##negate x) x))
                  (y-negative?
                   (##negative? y))
                  (abs-y
                   (if y-negative? (##negate y) y)))
             (if (##< abs-x abs-y)
               (##cons 0 x)
               (let ((result
                      (##bignum.div abs-x abs-y)))

                 (if (##not (##eq? x-negative? y-negative?))
                   (##set-car! result (##negate (##car result))))

                 (if x-negative?
                   (##set-cdr! result (##negate (##cdr result))))

                 result)))))))

(define-prim (##exact-int.nth-root x y)
  (cond ((##eq? x 0)
         0)
        ((##eq? y 1)
         x)
        ((or (##eq? x 1)
             (##not (##fixnum? y)))
         1)
        ((##fixnum.even? y)
         (##exact-int.nth-root (##car (##exact-int.sqrt x))
                               (##fixnum.arithmetic-shift-right y 1)))
        (else
         (let* ((length
                 (##integer-length x))
                (length/y/2 ; length/y/2 >= 0 (because x >= 1)
                 (##fixnum.arithmetic-shift-right
                  (##fixnum.quotient
                   (##fixnum.- length 1)
                   y)
                  1)))
           (if (##fixnum.zero? length/y/2)
             (if (##fixnum.< length y)
               1
               2)
             (let ((init-g
                    (let* ((length/y/2*y
                            (##fixnum.* length/y/2 y)))
                      (if (##fixnum.zero? (##fixnum.arithmetic-shift-right
                                           (##fixnum.quotient
                                            (##fixnum.- length
                                                        length/y/2*y
                                                        1)
                                            y)
                                           1))
                        ; here the recursion would yield 1 or 2 as the
                        ; result, which isn't good so we use a better
                        ; initial guess.
                        (##fixnum.arithmetic-shift-left
                         1
                         (##fixnum.quotient
                          (##fixnum.+ length (##fixnum.- y 1))
                          y))
                        (let* ((top-bits
                                (##arithmetic-shift
                                 x
                                 (##fixnum.- length/y/2*y)))
                               (nth-root-top-bits
                                (##exact-int.nth-root top-bits y)))
                          (##arithmetic-shift
                           (##+ nth-root-top-bits 1)
                           length/y/2))))))
               (let loop ((g init-g))
                 (let* ((a (##expt g (##fixnum.- y 1)))
                        (b (##* a y))
                        (c (##* a (##fixnum.- y 1)))
                        (d (##quotient (##+ x (##* g c)) b)))
                   (let ((diff (##- d g)))
                     (cond ((##not (##negative? diff))
                            g)
                           ((##< diff -1)
                            (loop d))
                           (else
                            ; once the difference is one, it's more
                            ; efficient to just decrement until g^y <= x
                            (let loop ((g d))
                              (if (##not (##< x (##expt g y)))
                                g
                                (loop (##- g 1)))))))))))))))

(define-prim (##integer-nth-root x y)

  (define (type-error-on-x)
    (##fail-check-exact-integer 1 integer-nth-root x y))

  (define (type-error-on-y)
    (##fail-check-exact-integer 2 integer-nth-root x y))

  (define (range-error-on-x)
    (##raise-range-exception 1 integer-nth-root x y))

  (define (range-error-on-y)
    (##raise-range-exception 2 integer-nth-root x y))

  (if (macro-exact-int? x)
    (if (macro-exact-int? y)
      (cond ((##negative? x)
             (range-error-on-x))
            ((##positive? y)
             (##exact-int.nth-root x y))
            (else
             (range-error-on-y)))
      (type-error-on-y))
    (type-error-on-x)))

(define-prim (integer-nth-root x y)
  (macro-force-vars (x y)
    (##integer-nth-root x y)))

(define-prim (##exact-int.sqrt x)

  ;; Derived from the paper "Karatsuba Square Root" by Paul Zimmermann,
  ;; INRIA technical report RR-3805, 1999.  (Used in gmp 4.*)

  ;; Note that the statement of the theorem requires that
  ;; b/4 <= high-order digit of x < b which can be impossible when b is a
  ;; power of 2; the paper later notes that it is the lower bound that is
  ;; necessary, which we preserve.

  (if (and (##fixnum? x)
           ;; we require that
           ;; (##< (##flonum.sqrt (- (* y y) 1)) y) => #t
           ;; whenever x=y^2 is in this range.  Here we assume that we
           ;; have at least as much precision as IEEE double precision and
           ;; we round to nearest.
           (or (##not (##fixnum? 4294967296))          ; 32-bit fixnums
               (##fixnum.<= x 4503599627370496)))      ; 2^52
      (let* ((s (##flonum.->fixnum (##flonum.sqrt (##flonum.<-fixnum x))))
             (r (##fixnum.- x (##fixnum.* s s))))
        (##cons s r))
      (let ((length/4
             (##fixnum.arithmetic-shift-right
              (##fixnum.+ (##integer-length x) 1)
              2)))
        (let* ((s-prime&r-prime
                (##exact-int.sqrt
                 (##arithmetic-shift
                  x
                  (##fixnum.- (##fixnum.arithmetic-shift-left length/4 1)))))
               (s-prime
                (##car s-prime&r-prime))
               (r-prime
                (##cdr s-prime&r-prime)))
          (let* ((qu
                  (##exact-int.div
                   (##+ (##arithmetic-shift r-prime length/4)
                        (##extract-bit-field length/4 length/4 x))
                   (##arithmetic-shift s-prime 1)))
                 (q
                  (##car qu))
                 (u
                  (##cdr qu)))
            (let ((s
                   (##+ (##arithmetic-shift s-prime length/4) q))
                  (r
                   (##- (##+ (##arithmetic-shift u length/4)
                             (##extract-bit-field length/4 0 x))
                        (##* q q))))
              (if (##negative? r)
                (##cons (##- s 1)
                        (##+ r
                             (##- (##arithmetic-shift s 1) 1)))
                (##cons s
                        r))))))))

(define-prim (##integer-sqrt x)

  (define (type-error)
    (##fail-check-exact-integer 1 integer-sqrt x))

  (define (range-error)
    (##raise-range-exception 1 integer-sqrt x))

  (if (macro-exact-int? x)
    (if (##negative? x)
      (range-error)
      (##car (##exact-int.sqrt x)))
    (type-error)))

(define-prim (integer-sqrt x)
  (macro-force-vars (x)
    (##integer-sqrt x)))

(define-prim (##exact-int.square n)
  (##* n n))

;------------------------------------------------------------------------------

; Ratnum operations
; -----------------

(define-prim (##ratnum.= x y)
  (and (##= (macro-ratnum-numerator x)
            (macro-ratnum-numerator y))
       (##= (macro-ratnum-denominator x)
            (macro-ratnum-denominator y))))

(define-prim (##ratnum.< x y)
  (##< (##* (macro-ratnum-numerator x)
            (macro-ratnum-denominator y))
       (##* (macro-ratnum-denominator x)
            (macro-ratnum-numerator y))))

(define-prim (##ratnum.+ x y)
  (let ((p (macro-ratnum-numerator x))
        (q (macro-ratnum-denominator x))
        (r (macro-ratnum-numerator y))
        (s (macro-ratnum-denominator y)))
    (let ((d1 (##gcd q s)))
      (if (##eq? d1 1)
        (macro-ratnum-make (##+ (##* p s)
                                (##* r q))
                           (##* q s))
        (let* ((s-prime (##quotient s d1))
               (t (##+ (##* p s-prime)
                       (##* r (##quotient q d1))))
               (d2 (##gcd d1 t))
               (num (##quotient t d2))
               (den (##* (##quotient q d2)
                         s-prime)))
          (if (##eq? den 1)
            num
            (macro-ratnum-make num den)))))))

(define-prim (##ratnum.- x y)
  (let ((p (macro-ratnum-numerator x))
        (q (macro-ratnum-denominator x))
        (r (macro-ratnum-numerator y))
        (s (macro-ratnum-denominator y)))
    (let ((d1 (##gcd q s)))
      (if (##eq? d1 1)
        (macro-ratnum-make (##- (##* p s)
                                (##* r q))
                           (##* q s))
        (let* ((s-prime (##quotient s d1))
               (t (##- (##* p s-prime)
                       (##* r (##quotient q d1))))
               (d2 (##gcd d1 t))
               (num (##quotient t d2))
               (den (##* (##quotient q d2)
                         s-prime)))
          (if (##eq? den 1)
            num
            (macro-ratnum-make num den)))))))

(define-prim (##ratnum.* x y)
  (let ((p (macro-ratnum-numerator x))
        (q (macro-ratnum-denominator x))
        (r (macro-ratnum-numerator y))
        (s (macro-ratnum-denominator y)))
    (if (##eq? x y)
      (macro-ratnum-make (##* p p) (##* q q))     ; already in lowest form
      (let* ((gcd-ps (##gcd p s))
             (gcd-rq (##gcd r q))
             (num (##* (##quotient p gcd-ps) (##quotient r gcd-rq)))
             (den (##* (##quotient q gcd-rq) (##quotient s gcd-ps))))
        (if (##eq? den 1)
          num
          (macro-ratnum-make num den))))))

(define-prim (##ratnum./ x y)
  (let ((p (macro-ratnum-numerator x))
        (q (macro-ratnum-denominator x))
        (r (macro-ratnum-denominator y))
        (s (macro-ratnum-numerator y)))
    (if (##eq? x y)
      1
      (let* ((gcd-ps (##gcd p s))
             (gcd-rq (##gcd r q))
             (num (##* (##quotient p gcd-ps) (##quotient r gcd-rq)))
             (den (##* (##quotient q gcd-rq) (##quotient s gcd-ps))))
        (if (##negative? den)
          (if (##eq? den -1)
            (##negate num)
            (macro-ratnum-make (##negate num) (##negate den)))
          (if (##eq? den 1)
            num
            (macro-ratnum-make num den)))))))

(define-prim (##ratnum.normalize num den)
  (let* ((x (##gcd num den))
         (y (if (##negative? den) (##negate x) x))
         (num (##quotient num y))
         (den (##quotient den y)))
    (if (##eq? den 1)
      num
      (macro-ratnum-make num den))))

(define-prim (##ratnum.<-exact-int x)
  (macro-ratnum-make x 1))

;------------------------------------------------------------------------------

; Flonum operations
; -----------------

(define-prim (##flonum.->fixnum x))
(define-prim (##flonum.<-fixnum x))

(define-prim-nary (##flonum.max x y)
  ()
  x
  (##flonum.max x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##flonum.min x y)
  ()
  x
  (##flonum.min x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##flonum.+ x y)
  (macro-inexact-+0)
  x
  (##flonum.+ x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##flonum.* x y)
  (macro-inexact-+1)
  x
  (##flonum.* x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##flonum.- x y)
  ()
  (##flonum.- x)
  (##flonum.- x y)
  macro-no-force
  macro-no-check)

(define-prim-nary (##flonum./ x y)
  ()
  (##flonum./ (macro-inexact-+1) x)
  (##flonum./ x y)
  macro-no-force
  macro-no-check)

(define-prim (##flonum.abs x))

(define-prim (##flonum.floor x))
(define-prim (##flonum.ceiling x))

(define-prim (##flonum.truncate x)
  (let ((y (##flonum.floor x)))
    (if (and (##flonum.negative? x) (##not (##flonum.= x y)))
      (##flonum.+ y (macro-inexact-+1))
      y)))

(define-prim (##flonum.round x))

(define-prim (##flonum.exp x))
(define-prim (##flonum.log x))
(define-prim (##flonum.sin x))
(define-prim (##flonum.cos x))
(define-prim (##flonum.tan x))
(define-prim (##flonum.asin x))
(define-prim (##flonum.acos x))
(define-prim (##flonum.atan x #!optional (y (macro-absent-obj)))
  (if (##eq? y (macro-absent-obj))
    (##flonum.atan x)
    (##flonum.atan x y)))
(define-prim (##flonum.sqrt x))
(define-prim (##flonum.copysign x y))

(define-prim (##flonum.zero? x)
  (##flonum.= x (macro-inexact-+0)))

(define-prim (##flonum.positive? x)
  (##flonum.< (macro-inexact-+0) x))

(define-prim (##flonum.negative? x)
  (##flonum.< x (macro-inexact-+0)))

(define-prim (##flonum.finite? x)
  (##not (or (##flonum.nan? x)
             (##flonum.= x +inf.)
             (##flonum.= x -inf.))))

(define-prim (##flonum.nan? x))

(define-prim (##flonum.<-fixnum-exact? x))

(define-prim-nary-bool (##flonum.= x y)
  #t
  #t
  (##flonum.= x y)
  macro-no-force
  macro-no-check)

(define-prim-nary-bool (##flonum.< x y)
  #t
  #t
  (##flonum.< x y)
  macro-no-force
  macro-no-check)

(define-prim-nary-bool (##flonum.> x y)
  #t
  #t
  (##flonum.> x y)
  macro-no-force
  macro-no-check)

(define-prim-nary-bool (##flonum.<= x y)
  #t
  #t
  (##flonum.<= x y)
  macro-no-force
  macro-no-check)

(define-prim-nary-bool (##flonum.>= x y)
  #t
  #t
  (##flonum.>= x y)
  macro-no-force
  macro-no-check)

(define-prim (##flonum.<-ratnum x)
  (let* ((num (macro-ratnum-numerator x))
         (n (##abs num))
         (d (macro-ratnum-denominator x))
         (wn (##integer-length n)) ; 2^(wn-1) <= n < 2^wn
         (wd (##integer-length d)) ; 2^(wd-1) <= d < 2^wd
         (p (##fixnum.- wn wd)))

    (define (f1 sn sd)
      (if (##< sn sd) ; n/(d*2^p) < 1 ?
        (f2 (##arithmetic-shift sn 1) sd (##fixnum.- p 1))
        (f2 sn sd p)))

    (define (f2 a b p)
      ; 1 <= a/b < 2  and  n/d = (2^p*a)/b  and  n/d < 2^(p+1)
      (let* ((shift
              (##fixnum.min (macro-flonum-m-bits)
                            (##fixnum.- p (macro-flonum-e-min))))
             (abs-result
              (##flonum.*
               (##flonum.<-exact-int
                (##round
                 (##ratnum.normalize
                  (##arithmetic-shift a shift)
                  b)))
               (##flonum.expt2 (##fixnum.- p shift)))))
        (if (##negative? num)
          (##flonum.copysign abs-result (macro-inexact--1))
          abs-result)))

    ; 2^(p-1) <= n/d < 2^(p+1)
    ; 1/2 <= n/(d*2^p) < 2  or equivalently  1/2 <= (n*2^-p)/d < 2

    (if (##fixnum.negative? p)
      (f1 (##arithmetic-shift n (##fixnum.- p)) d)
      (f1 n (##arithmetic-shift d p)))))

(define-prim (##flonum.<-exact-int x)

  (define (f1 x)
    (let* ((w ; 2^(w-1) <= x < 2^w
            (##integer-length x))
           (p ; 2^52 <= x/2^p < 2^53
            (##fixnum.- w (macro-flonum-m-bits-plus-1))))
      (if (##fixnum.< p 1)
        (f2 x)
        (let* ((2^p (##arithmetic-shift 1 p))
               (qr (##exact-int.div x 2^p))
               (q (##car qr))
               (r (##cdr qr))
               (r*2 (##arithmetic-shift r 1)))
          (##flonum.*
           (##flonum.expt2 p)
           (f2 (if (or (and (##not (##odd? q))
                            (##= r*2 2^p))
                       (##< r*2 2^p))
                 q
                 (##+ q 1))))))))

  (define (f2 x) ; 0 <= x < 2^53
    (if (##fixnum? x)
      (##flonum.<-fixnum x)
      (let* ((x (if (##fixnum? x) (##bignum.<-fixnum x) x))
             (n (##bignum.mdigit-length x)))
        (let loop ((i (##fixnum.- n 1))
                   (result (macro-inexact-+0)))
          (if (##fixnum.< i 0)
            result
            (let ((mdigit (##bignum.mdigit-ref x i)))
              (loop (##fixnum.- i 1)
                    (##flonum.+ (##flonum.* result
                                            ##bignum.inexact-mdigit-base)
                                (##flonum.<-fixnum mdigit)))))))))

  (if (##fixnum? x)
    (##flonum.<-fixnum x)
    (if (##negative? x)
      (##flonum.copysign (f1 (##negate x)) (macro-inexact--1))
      (f1 x))))

(define-prim (##flonum.expt2 n)
  (cond ((##fixnum.zero? n)
         (macro-inexact-+1))
        ((##fixnum.negative? n)
         (##expt (macro-inexact-+1/2) (##fixnum.- n)))
        (else
         (##expt (macro-inexact-+2) n))))

(define-prim (##flonum.->exact-int x)
  (let loop1 ((z (##flonum.abs x))
              (n 1))
    (if (##flonum.< ##bignum.inexact-mdigit-base z)
      (loop1 (##flonum./ z ##bignum.inexact-mdigit-base)
             (##fixnum.+ n 1))
      (let loop2 ((result 0)
                  (z z)
                  (i n))
        (if (##fixnum.< 0 i)
          (let* ((inexact-floor-z
                  (##flonum.floor z))
                 (floor-z
                  (##flonum.->fixnum inexact-floor-z))
                 (new-z
                  (##flonum.* (##flonum.- z inexact-floor-z)
                              ##bignum.inexact-mdigit-base)))
            (loop2 (##+ floor-z
                        (##arithmetic-shift result ##bignum.mdigit-width))
                   new-z
                   (##fixnum.- i 1)))
          (if (##flonum.negative? x)
            (##negate result)
            result))))))

(define-prim (##flonum.->inexact-exponential-format x)

  (define (exp-form-pos x y i)
    (let ((i*2 (##fixnum.+ i i)))
      (let ((z (if (and (##not (##fixnum.< (macro-flonum-e-bias) i*2))
                        (##not (##flonum.< x y)))
                 (exp-form-pos x (##flonum.* y y) i*2)
                 (##vector x 0 1))))
        (let ((a (##vector-ref z 0)) (b (##vector-ref z 1)))
          (let ((i+b (##fixnum.+ i b)))
            (if (and (##not (##fixnum.< (macro-flonum-e-bias) i+b))
                     (##not (##flonum.< a y)))
              (begin
                (##vector-set! z 0 (##flonum./ a y))
                (##vector-set! z 1 i+b)))
            z)))))

  (define (exp-form-neg x y i)
    (let ((i*2 (##fixnum.+ i i)))
      (let ((z (if (and (##fixnum.< i*2 (macro-flonum-e-bias-minus-1))
                        (##flonum.< x y))
                 (exp-form-neg x (##flonum.* y y) i*2)
                 (##vector x 0 1))))
        (let ((a (##vector-ref z 0)) (b (##vector-ref z 1)))
          (let ((i+b (##fixnum.+ i b)))
            (if (and (##fixnum.< i+b (macro-flonum-e-bias-minus-1))
                     (##flonum.< a y))
              (begin
                (##vector-set! z 0 (##flonum./ a y))
                (##vector-set! z 1 i+b)))
            z)))))

  (define (exp-form x)
    (if (##flonum.< x (macro-inexact-+1))
      (let ((z (exp-form-neg x (macro-inexact-+1/2) 1)))
        (##vector-set! z 0 (##flonum.* (macro-inexact-+2) (##vector-ref z 0)))
        (##vector-set! z 1 (##fixnum.- -1 (##vector-ref z 1)))
        z)
      (exp-form-pos x (macro-inexact-+2) 1)))

  (if (##flonum.negative? (##flonum.copysign (macro-inexact-+1) x))
    (let ((z (exp-form (##flonum.copysign x (macro-inexact-+1)))))
      (##vector-set! z 2 -1)
      z)
    (exp-form x)))

(define-prim (##flonum.->exact-exponential-format x)
  (let ((z (##flonum.->inexact-exponential-format x)))
    (let ((y (##vector-ref z 0)))
      (if (##not (##flonum.< y (macro-inexact-+2))) ; +inf. or +nan.?
        (begin
          (if (##flonum.< (macro-inexact-+0) y)
            (##vector-set! z 0 (macro-flonum-+m-min))  ; +inf.
            (##vector-set! z 0 (macro-flonum-+m-max))) ; +nan.
          (##vector-set! z 1 (macro-flonum-e-bias-plus-1)))
        (##vector-set! z 0
          (##flonum.->exact-int
           (##flonum.* (##vector-ref z 0) (macro-flonum-m-min)))))
      (##vector-set! z 1 (##fixnum.- (##vector-ref z 1) (macro-flonum-m-bits)))
      z)))

(define-prim (##flonum.inexact->exact x)
  (let ((y (##flonum.->exact-exponential-format x)))
    (##exact-int.*-expt2
     (if (##fixnum.negative? (##vector-ref y 2))
       (##negate (##vector-ref y 0))
       (##vector-ref y 0))
     (##vector-ref y 1))))

(define-prim (##flonum.->ratnum x)
  (let ((y (##flonum.inexact->exact x)))
    (if (macro-exact-int? y)
      (##ratnum.<-exact-int y)
      y)))

(define-prim (##flonum.->ieee754-32 x)
  (##u32vector-ref (##f32vector x) 0))

(define-prim (##flonum.<-ieee754-32 n)
  (let ((x (##u32vector n)))
    (##f32vector-ref x 0)))

(define-prim (##flonum.->ieee754-64 x)
  (##u64vector-ref x 0))

(define-prim (##flonum.<-ieee754-64 n)
  (let ((x (##u64vector n)))
    (##subtype-set! x (macro-subtype-flonum))
    x))

;------------------------------------------------------------------------------

; Cpxnum operations
; -----------------

(define-prim (##cpxnum.= x y)
  (and (##= (macro-cpxnum-real x) (macro-cpxnum-real y))
       (##= (macro-cpxnum-imag x) (macro-cpxnum-imag y))))

(define-prim (##cpxnum.+ x y)
  (let ((a (macro-cpxnum-real x)) (b (macro-cpxnum-imag x))
        (c (macro-cpxnum-real y)) (d (macro-cpxnum-imag y)))
    (##make-rectangular (##+ a c) (##+ b d))))

(define-prim (##cpxnum.* x y)
  (let ((a (macro-cpxnum-real x)) (b (macro-cpxnum-imag x))
        (c (macro-cpxnum-real y)) (d (macro-cpxnum-imag y)))
    (##make-rectangular (##- (##* a c) (##* b d)) (##+ (##* a d) (##* b c)))))

(define-prim (##cpxnum.- x y)
  (let ((a (macro-cpxnum-real x)) (b (macro-cpxnum-imag x))
        (c (macro-cpxnum-real y)) (d (macro-cpxnum-imag y)))
    (##make-rectangular (##- a c) (##- b d))))

(define-prim (##cpxnum./ x y)

  (define (basic/ a b c d q)
    (##make-rectangular (##/ (##+ (##* a c) (##* b d)) q)
                        (##/ (##- (##* b c) (##* a d)) q)))

  (let ((a (macro-cpxnum-real x)) (b (macro-cpxnum-imag x))
        (c (macro-cpxnum-real y)) (d (macro-cpxnum-imag y)))
    (cond ((##eq? d 0)
           ; A normalized cpxnum can't have an imaginary part that is
           ; exact 0 but it is possible that ##cpxnum./ receives a
           ; nonnormalized cpxnum as x or y when it is called from ##/.
           (##make-rectangular (##/ a c)
                               (##/ b c)))
          ((##eq? c 0)
           (##make-rectangular (##/ b d)
                               (##negate (##/ a d))))
          ((and (##exact? c) (##exact? d))
           (basic/ a b c d (##+ (##* c c) (##* d d))))
          (else
           ; just coerce everything to inexact and move on
           (let ((inexact-c (##exact->inexact c))
                 (inexact-d (##exact->inexact d)))
             (if (and (##flonum.finite? inexact-c)
                      (##flonum.finite? inexact-d))
               (let ((q
                      (##flonum.+ (##flonum.* inexact-c inexact-c)
                                  (##flonum.* inexact-d inexact-d))))
                 (cond ((##not (##flonum.finite? q))
                        (let ((a
                               (if (##flonum? a)
                                 (##flonum.* a (macro-inexact-scale-down))
                                 (##* a (macro-scale-down))))
                              (b
                               (if (##flonum? b)
                                 (##flonum.* b (macro-inexact-scale-down))
                                 (##* b (macro-scale-down))))
                              (inexact-c
                               (##flonum.* inexact-c
                                           (macro-inexact-scale-down)))
                              (inexact-d
                               (##flonum.* inexact-d
                                           (macro-inexact-scale-down))))
                          (basic/ a
                                  b
                                  inexact-c
                                  inexact-d
                                  (##flonum.+
                                   (##flonum.* inexact-c inexact-c)
                                   (##flonum.* inexact-d inexact-d)))))
                       ((##flonum.< q (macro-flonum-min-normal))
                        (let ((a
                               (if (##flonum? a)
                                 (##flonum.* a (macro-inexact-scale-up))
                                 (##* a (macro-scale-up))))
                              (b
                               (if (##flonum? b)
                                 (##flonum.* b (macro-inexact-scale-up))
                                 (##* b (macro-scale-up))))
                              (inexact-c
                               (##flonum.* inexact-c
                                           (macro-inexact-scale-up)))
                              (inexact-d
                               (##flonum.* inexact-d
                                           (macro-inexact-scale-up))))
                          (basic/ a
                                  b
                                  inexact-c
                                  inexact-d
                                  (##flonum.+
                                   (##flonum.* inexact-c inexact-c)
                                   (##flonum.* inexact-d inexact-d)))))
                       (else
                        (basic/ a b inexact-c inexact-d q))))
               (cond ((##flonum.= inexact-c (macro-inexact-+inf))
                      (basic/ a
                              b
                              (macro-inexact-+0)
                              (if (##flonum.nan? inexact-d)
                                inexact-d
                                (##flonum.copysign (macro-inexact-+0)
                                                   inexact-d))
                              (macro-inexact-+1)))
                     ((##flonum.= inexact-c (macro-inexact--inf))
                      (basic/ a
                              b
                              (macro-inexact--0)
                              (if (##flonum.nan? inexact-d)
                                inexact-d
                                (##flonum.copysign (macro-inexact-+0)
                                                   inexact-d))
                              (macro-inexact-+1)))
                     ((##flonum.nan? inexact-c)
                      (cond ((##flonum.= inexact-d (macro-inexact-+inf))
                             (basic/ a
                                     b
                                     inexact-c
                                     (macro-inexact-+0)
                                     (macro-inexact-+1)))
                            ((##flonum.= inexact-d (macro-inexact--inf))
                             (basic/ a
                                     b
                                     inexact-c
                                     (macro-inexact--0)
                                     (macro-inexact-+1)))
                            ((##flonum.nan? inexact-d)
                             (basic/ a
                                     b
                                     inexact-c
                                     inexact-d
                                     (macro-inexact-+1)))
                            (else
                             (basic/ a
                                     b
                                     inexact-c
                                     (macro-inexact-+nan)
                                     (macro-inexact-+1)))))
                     (else
                      ; finite inexact-c
                      (cond ((##flonum.nan? inexact-d)
                             (basic/ a
                                     b
                                     (macro-inexact-+nan)
                                     inexact-d
                                     (macro-inexact-+1)))
                            (else
                             ; inexact-d is +inf. or -inf.
                             (basic/ a
                                     b
                                     (##flonum.copysign (macro-inexact-+0)
                                                        inexact-c)
                                     (##flonum.copysign (macro-inexact-+0)
                                                        inexact-d)
                                     (macro-inexact-+1))))))))))))

(define-prim (##cpxnum.<-noncpxnum x)
  (macro-cpxnum-make x 0))

;------------------------------------------------------------------------------

; Pseudo-random number generation, compatible with srfi-27.

; This code is based on Pierre Lecuyer's MRG32K3A generator.

(define-type random-source
  id: 1b002758-f900-4e96-be5e-fa407e331fc0
  implementer: implement-type-random-source
  constructor: macro-make-random-source
  type-exhibitor: macro-type-random-source
  macros:
  prefix: macro-
  opaque:

  (state-ref         unprintable: read-only:)
  (state-set!        unprintable: read-only:)
  (randomize!        unprintable: read-only:)
  (pseudo-randomize! unprintable: read-only:)
  (make-integers     unprintable: read-only:)
  (make-reals        unprintable: read-only:)
)

(define-check-type random-source
  (macro-type-random-source)
  macro-random-source?)

(implement-type-random-source)
(implement-check-type-random-source)

(define-prim (##make-random-source-mrg32k3a)

  (##define-macro (macro-w)
    65536)

  (##define-macro (macro-w^2-mod-m1)
    209)

  (##define-macro (macro-w^2-mod-m2)
    22853)

  (##define-macro (macro-m1)
    4294967087) ; (- (expt (macro-w) 2) (macro-w^2-mod-m1))

  (##define-macro (macro-m1-inexact)
    4294967087.0) ; (exact->inexact (macro-m1))

  (##define-macro (macro-m1-plus-1-inexact)
    4294967088.0) ; (exact->inexact (+ (macro-m1) 1))

  (##define-macro (macro-inv-m1-plus-1-inexact)
    2.328306549295728e-10) ; (exact->inexact (/ (+ (macro-m1) 1)))

  (##define-macro (macro-m1-minus-1)
    4294967086) ; (- (macro-m1) 1)

  (##define-macro (macro-k)
    28)

  (##define-macro (macro-2^k)
    268435456) ; (expt 2 (macro-k))

  (##define-macro (macro-2^k-inexact)
    268435456.0) ; (exact->inexact (expt 2 (macro-k)))

  (##define-macro (macro-inv-2^k-inexact)
    3.725290298461914e-9) ; (exact->inexact (/ (expt 2 (macro-k))))

  (##define-macro (macro-2^53-k-inexact)
    33554432.0) ; (exact->inexact (expt 2 (- 53 (macro-k))))

  (##define-macro (macro-m1-div-2^k-inexact)
    15.0) ; (exact->inexact (quotient (macro-m1) (expt 2 (macro-k))))

  (##define-macro (macro-m1-div-2^k-times-2^k-inexact)
    4026531840.0) ; (exact->inexact (* (quotient (macro-m1) (expt 2 (macro-k))) (expt 2 (macro-k))))

  (##define-macro (macro-m2)
    4294944443) ; (- (expt (macro-w) 2) (macro-w^2-mod-m2))

  (##define-macro (macro-m2-inexact)
    4294944443.0) ; (exact->inexact (macro-m2))

  (##define-macro (macro-m2-minus-1)
    4294944442) ; (- (macro-m2) 1)

  (define (pack-state a b c d e f)
    (##f64vector
     (##flonum.<-exact-int a)
     (##flonum.<-exact-int b)
     (##flonum.<-exact-int c)
     (##flonum.<-exact-int d)
     (##flonum.<-exact-int e)
     (##flonum.<-exact-int f)
     0.0 ; where the result of advance-state! is put
     0.0 ; q in rand-fixnum32
     0.0 ; qn in rand-fixnum32
     ))

  (define (unpack-state state)
    (##vector
     (##flonum.->exact-int (##f64vector-ref state 0))
     (##flonum.->exact-int (##f64vector-ref state 1))
     (##flonum.->exact-int (##f64vector-ref state 2))
     (##flonum.->exact-int (##f64vector-ref state 3))
     (##flonum.->exact-int (##f64vector-ref state 4))
     (##flonum.->exact-int (##f64vector-ref state 5))))

  (let ((state ; initial state is 0 3 6 9 12 15 of A^(2^4), see below
         (pack-state
          1062452522
          2961816100
           342112271
          2854655037
          3321940838
          3542344109)))

    (define (state-ref)
      (unpack-state state))

    (define (state-set! rs new-state)

      (define (integer-in-range? x m)
        (and (macro-exact-int? x)
             (##not (##negative? x))
             (##< x m)))

      (or (and (##vector? new-state)
               (##fixnum.= (##vector-length new-state) 6)
               (let ((a (##vector-ref new-state 0))
                     (b (##vector-ref new-state 1))
                     (c (##vector-ref new-state 2))
                     (d (##vector-ref new-state 3))
                     (e (##vector-ref new-state 4))
                     (f (##vector-ref new-state 5)))
                 (and (integer-in-range? a (macro-m1))
                      (integer-in-range? b (macro-m1))
                      (integer-in-range? c (macro-m1))
                      (integer-in-range? d (macro-m2))
                      (integer-in-range? e (macro-m2))
                      (integer-in-range? f (macro-m2))
                      (##not (and (##eq? a 0) (##eq? b 0) (##eq? c 0)))
                      (##not (and (##eq? d 0) (##eq? e 0) (##eq? f 0)))
                      (begin
                        (set! state
                          (pack-state a b c d e f))
                        (##void)))))
          (##raise-type-exception
           2
           'random-source-state
           random-source-state-set!
           (##list rs new-state))))

    (define (randomize!)

      (define (random-fixnum-from-time)
        (let ((v (##f64vector 0.0)))
          (##get-current-time! v)
          (let ((x (##f64vector-ref v 0)))
            (##flonum.->fixnum
             (##flonum.* 536870912.0 ; (expt 2.0 29)
                         (##flonum.- x (##flonum.floor x)))))))

      (define seed16
        (random-fixnum-from-time))

      (define (simple-random16)
        (let ((r (##bitwise-and seed16 65535)))
          (set! seed16
            (##+ (##* 30903 r)
                 (##arithmetic-shift seed16 -16)))
          r))

      (define (simple-random32)
        (##+ (##arithmetic-shift (simple-random16) 16)
             (simple-random16)))

      ; perturb the state randomly

      (let ((s (unpack-state state)))
        (set! state
          (pack-state
           (##+ 1
                (##modulo (##+ (##vector-ref s 0)
                               (simple-random32))
                          (macro-m1-minus-1)))
           (##modulo (##+ (##vector-ref s 1)
                          (simple-random32))
                     (macro-m1))
           (##modulo (##+ (##vector-ref s 2)
                          (simple-random32))
                     (macro-m1))
           (##+ 1
                (##modulo (##+ (##vector-ref s 3)
                               (simple-random32))
                          (macro-m2-minus-1)))
           (##modulo (##+ (##vector-ref s 4)
                          (simple-random32))
                     (macro-m2))
           (##modulo (##+ (##vector-ref s 5)
                          (simple-random32))
                     (macro-m2))))
        (##void)))

    (define (pseudo-randomize! i j)

      (define (mult A B) ; A*B

        (define (lc i0 i1 i2 j0 j1 j2 m)
          (##modulo (##+ (##* (##vector-ref A i0)
                              (##vector-ref B j0))
                         (##+ (##* (##vector-ref A i1)
                                   (##vector-ref B j1))
                              (##* (##vector-ref A i2)
                                   (##vector-ref B j2))))
                    m))

        (##vector
         (lc  0  1  2   0  3  6  (macro-m1))
         (lc  0  1  2   1  4  7  (macro-m1))
         (lc  0  1  2   2  5  8  (macro-m1))
         (lc  3  4  5   0  3  6  (macro-m1))
         (lc  3  4  5   1  4  7  (macro-m1))
         (lc  3  4  5   2  5  8  (macro-m1))
         (lc  6  7  8   0  3  6  (macro-m1))
         (lc  6  7  8   1  4  7  (macro-m1))
         (lc  6  7  8   2  5  8  (macro-m1))
         (lc  9 10 11   9 12 15  (macro-m2))
         (lc  9 10 11  10 13 16  (macro-m2))
         (lc  9 10 11  11 14 17  (macro-m2))
         (lc 12 13 14   9 12 15  (macro-m2))
         (lc 12 13 14  10 13 16  (macro-m2))
         (lc 12 13 14  11 14 17  (macro-m2))
         (lc 15 16 17   9 12 15  (macro-m2))
         (lc 15 16 17  10 13 16  (macro-m2))
         (lc 15 16 17  11 14 17  (macro-m2))))

      (define (power A e) ; A^e
        (cond ((##eq? e 0)
               identity)
              ((##eq? e 1)
               A)
              ((##even? e)
               (power (mult A A) (##arithmetic-shift e -1)))
              (else
               (mult (power A (##- e 1)) A))))

      (define identity
        '#(         1           0           0
                    0           1           0
                    0           0           1
                    1           0           0
                    0           1           0
                    0           0           1))

      (define A ; primary MRG32k3a equations
        '#(         0     1403580  4294156359
                    1           0           0
                    0           1           0
               527612           0  4293573854
                    1           0           0
                    0           1           0))

      (define A^2^127 ; A^(2^127)
        '#(1230515664   986791581  1988835001
           3580155704  1230515664   226153695
            949770784  3580155704  2427906178
           2093834863    32183930  2824425944
           1022607788  1464411153    32183930
           1610723613   277697599  1464411153))

      (define A^2^76 ; A^(2^76)
        '#(  69195019  3528743235  3672091415
           1871391091    69195019  3672831523
           4127413238  1871391091    82758667
           3708466080  4292754251  3859662829
           3889917532  1511326704  4292754251
           1610795712  3759209742  1511326704))

      (define A^2^4 ; A^(2^4)
        '#(1062452522   340793741  2955879160
           2961816100  1062452522   387300998
            342112271  2961816100   736416029
           2854655037  1817134745  3493477402
           3321940838   818368950  1817134745
           3542344109  3790774567   818368950))

      (let ((M ; M = A^(2^4 + i*2^127 + j*2^76)
             (mult A^2^4
                   (mult (power A^2^127 i)
                         (power A^2^76 j)))))
        (set! state
          (pack-state
           (##vector-ref M 0)
           (##vector-ref M 3)
           (##vector-ref M 6)
           (##vector-ref M 9)
           (##vector-ref M 12)
           (##vector-ref M 15)))
        (##void)))

    (define (advance-state!)
      (##declare (not interrupts-enabled))
      (let* ((x10
              (##flonum.- (##flonum.* 1403580.0 (##f64vector-ref state 1))
                          (##flonum.* 810728.0 (##f64vector-ref state 2))))
             (y10
              (##flonum.- x10
                          (##flonum.* (##flonum.floor
                                       (##flonum./ x10 (macro-m1-inexact)))
                                      (macro-m1-inexact))))
             (x20
              (##flonum.- (##flonum.* 527612.0 (##f64vector-ref state 3))
                          (##flonum.* 1370589.0 (##f64vector-ref state 5))))
             (y20
              (##flonum.- x20
                          (##flonum.* (##flonum.floor
                                       (##flonum./ x20 (macro-m2-inexact)))
                                      (macro-m2-inexact))))
             (dx
              (##flonum.- y10 y20))
             (dy
              (##flonum.- dx
                          (##flonum.* (##flonum.floor
                                       (##flonum./ dx (macro-m1-inexact)))
                                      (macro-m1-inexact)))))
        (##f64vector-set! state 5 (##f64vector-ref state 4))
        (##f64vector-set! state 4 (##f64vector-ref state 3))
        (##f64vector-set! state 3 y20)
        (##f64vector-set! state 2 (##f64vector-ref state 1))
        (##f64vector-set! state 1 (##f64vector-ref state 0))
        (##f64vector-set! state 0 y10)
        (##f64vector-set! state 6 dy)))

    (define (make-integers)

      (define (random-integer range)

        (define (type-error)
          (##fail-check-exact-integer 1 random-integer range))

        (define (range-error)
          (##raise-range-exception 1 random-integer range))

        (macro-force-vars (range)
          (cond ((##fixnum? range)
                 (if (##fixnum.positive? range)
                   (if (##fixnum.< (macro-max-fixnum32) range)
                     (rand-integer range)
                     (rand-fixnum32 range))
                   (range-error)))
                ((##bignum? range)
                 (if (##bignum.negative? range)
                   (range-error)
                   (rand-integer range)))
                (else
                 (type-error)))))

      random-integer)

    (define (rand-integer range)

      ; constants for computing fixnum approximation of inverse of range

      (define size 14)
      (define 2^2*size 268435456)

      (let ((len (##integer-length range)))
        (if (##fixnum.= (##fixnum.- len 1) ; check if range is a power of 2
                        (##first-set-bit range))
          (rand-integer-2^ (##fixnum.- len 1))
          (let* ((inv
                  (##fixnum.quotient
                   2^2*size
                   (##fixnum.+
                    1
                    (##extract-bit-field size (##fixnum.- len size) range))))
                 (range2
                  (##* range inv)))
            (let loop ()
              (let ((r (rand-integer-2^ (##fixnum.+ len size))))
                (if (##< r range2)
                  (##quotient r inv)
                  (loop))))))))

    (define (rand-integer-2^ w)

      (define (rand w s)
        (cond ((##fixnum.< w (macro-k))
               (##fixnum.bitwise-and
                (rand-fixnum32-2^k)
                (##fixnum.- (##fixnum.arithmetic-shift-left 1 w) 1)))
              ((##fixnum.= w (macro-k))
               (rand-fixnum32-2^k))
              (else
               (let ((s/2 (##fixnum.arithmetic-shift-right s 1)))
                 (if (##fixnum.< s w)
                   (##+ (rand s s/2)
                        (##arithmetic-shift (rand (##fixnum.- w s) s/2) s))
                   (rand w s/2))))))

      (define (split w s)
        (let ((s*2 (##fixnum.* 2 s)))
          (if (##fixnum.< s*2 w)
            (split w s*2)
            s)))

      (rand w (split w (macro-k))))

    (define (rand-fixnum32-2^k)
      (##declare (not interrupts-enabled))
      (let loop ()
        (advance-state!)
        (if (##flonum.< (##f64vector-ref state 6)
                        (macro-m1-div-2^k-times-2^k-inexact))
          (##flonum.->fixnum
           (##flonum./ (##f64vector-ref state 6)
                       (macro-m1-div-2^k-inexact)))
          (loop))))

    (define (rand-fixnum32 range) ; range is a fixnum32
      (##declare (not interrupts-enabled))
      (let* ((a (##flonum.<-fixnum range))
             (b (##flonum.floor (##flonum./ (macro-m1-inexact) a))))
        (##f64vector-set! state 7 b)
        (##f64vector-set! state 8 (##flonum.* a b)))
      (let loop ()
        (advance-state!)
        (if (##flonum.< (##f64vector-ref state 6)
                        (##f64vector-ref state 8))
          (##flonum.->fixnum
           (##flonum./ (##f64vector-ref state 6)
                       (##f64vector-ref state 7)))
          (loop))))

    (define (make-reals precision)
      (if (##flonum.< precision (macro-inv-m1-plus-1-inexact))
        (lambda ()
          (let loop ((r (##flonum.<-fixnum (rand-fixnum32-2^k)))
                     (d (macro-inv-2^k-inexact)))
            (if (##flonum.< r (macro-flonum-+m-max-inexact))
              (loop (##flonum.+ (##flonum.* r (macro-2^k-inexact))
                                (##flonum.<-fixnum (rand-fixnum32-2^k)))
                    (##flonum.* d (macro-inv-2^k-inexact)))
              (##flonum.* r d))))
        (lambda ()
          (##declare (not interrupts-enabled))
          (advance-state!)
          (##flonum./ (##flonum.+ 1.0 (##f64vector-ref state 6))
                      (macro-m1-plus-1-inexact)))))

    (macro-make-random-source
     state-ref
     state-set!
     randomize!
     pseudo-randomize!
     make-integers
     make-reals)))

(define-prim (make-random-source)
  (##make-random-source-mrg32k3a))

(define-prim (random-source? obj)
  (macro-force-vars (obj)
    (macro-random-source? obj)))

(define-prim (##random-source-state-ref rs)
  ((macro-random-source-state-ref rs)))

(define-prim (random-source-state-ref rs)
  (macro-force-vars (rs)
    (macro-check-random-source rs 1 (random-source-state-ref rs)
      (##random-source-state-ref rs))))

(define-prim (##random-source-state-set! rs new-state)
  ((macro-random-source-state-set! rs) rs new-state))

(define-prim (random-source-state-set! rs new-state)
  (macro-force-vars (rs new-state)
    (macro-check-random-source rs 1 (random-source-state-set! rs new-state)
      (##random-source-state-set! rs new-state))))

(define-prim (##random-source-randomize! rs)
  ((macro-random-source-randomize! rs)))

(define-prim (random-source-randomize! rs)
  (macro-force-vars (rs)
    (macro-check-random-source rs 1 (random-source-randomize! rs)
      (##random-source-randomize! rs))))

(define-prim (##random-source-pseudo-randomize! rs i j)
  ((macro-random-source-pseudo-randomize! rs) i j))

(define-prim (random-source-pseudo-randomize! rs i j)
  (macro-force-vars (rs i j)
    (macro-check-random-source rs 1 (random-source-pseudo-randomize! rs i j)
      (if (##not (macro-exact-int? i))
        (##fail-check-exact-integer 2 random-source-pseudo-randomize! rs i j)
        (if (##not (macro-exact-int? j))
          (##fail-check-exact-integer 3 random-source-pseudo-randomize! rs i j)
          (if (##negative? i)
            (##raise-range-exception 2 random-source-pseudo-randomize! rs i j)
            (if (##negative? j)
              (##raise-range-exception 3 random-source-pseudo-randomize! rs i j)
              (##random-source-pseudo-randomize! rs i j))))))))

(define-prim (##random-source-make-integers rs)
  ((macro-random-source-make-integers rs)))

(define-prim (random-source-make-integers rs)
  (macro-force-vars (rs)
    (macro-check-random-source rs 1 (random-source-make-integers rs)
      (##random-source-make-integers rs))))

(define-prim (##random-source-make-reals rs #!optional (p (macro-absent-obj)))
  ((macro-random-source-make-reals rs)
   (if (##eq? p (macro-absent-obj))
     1.0
     p)))

(define-prim (random-source-make-reals rs #!optional (p (macro-absent-obj)))
  (macro-force-vars (rs p)
    (macro-check-random-source rs 1 (random-source-make-reals rs p)
      (if (##eq? p (macro-absent-obj))
        (##random-source-make-reals rs)
        (if (##rational? p)
          (let ((precision (macro-real->inexact p)))
            (if (and (##flonum.< 0.0 precision)
                     (##flonum.< precision 1.0))
              (##random-source-make-reals rs precision)
              (##raise-range-exception 2 random-source-make-reals rs p)))
          (##fail-check-finite-real 2 random-source-make-reals rs p))))))

(define default-random-source #f)
(set! default-random-source (##make-random-source-mrg32k3a))

(define random-integer
  (##random-source-make-integers default-random-source))

(define random-real
  (##random-source-make-reals default-random-source))

;==============================================================================
